/*:
 * @target MZ
 * @plugindesc Saves game data to a directory in the user's folder instead of the game directory (PC and Mac compatible).
 * @help
 * This plugin redirects the save files to a user directory, 
 * ensuring compatibility with both Windows and macOS.
 *
 * If the user directory is empty or does not exist, it will copy 
 * the save files from the game directory to the user directory on startup.
 * This ensures users who are using the game save folder will be able to
 * continue uninterrupted.
 *
 * The save files will be stored in:
 * - Windows: C:/Users/<YourUserName>/AppData/Local/<YourGameName>/
 * - macOS: /Users/<YourUserName>/Library/Application Support/<YourGameName>/
 *
 * No plugin commands are necessary.
 */

(() => {
    const path = require('path');
    const fs = require('fs');
    const os = require('os');

    const gameName = 'SotS'; 
    
   

    // Function to get the user directory based on the platform
    const getSaveDirectory = () => {
        let basePath;

        if (process.platform === 'win32') {
            basePath = path.join(os.homedir(), 'AppData', 'Local', gameName + "/");
        } else if (process.platform === 'darwin') {
            basePath = path.join(os.homedir(), 'Documents', gameName + "/");
        } else {
            basePath = path.join(os.homedir(), '.' + gameName);
        }

       // console.log("User Save Directory:", basePath); // Debugging line

        // Ensure the directory exists
        if (!fs.existsSync(basePath)) {
            fs.mkdirSync(basePath, { recursive: true });
            //console.log("Created User Save Directory"); // Debugging line
        }
       // console.log(basePath);
        return basePath;
    };

    // Function to get the game directory
    const getGameDirectory = () => {
        const gameDir = path.join(process.cwd(), 'save'); // Assuming save files are in a 'save' folder in the game directory
       // console.log("Game Save Directory:", gameDir); // Debugging line
        return gameDir;
    };

    // Function to copy files from one directory to another
    const copySaveFiles = (srcDir, destDir) => {
        const saveFiles = fs.readdirSync(srcDir).filter(file => file.endsWith('.rmmzsave'));
        saveFiles.forEach(file => {
            const srcPath = path.join(srcDir, file);
            const destPath = path.join(destDir, file);
            fs.copyFileSync(srcPath, destPath);
            //console.log(`Copied ${file} to ${destDir}`); // Debugging line
        });
    };

    // Function to check if the save directory is empty
    const isDirectoryEmpty = (dir) => {
        const isEmpty = !fs.existsSync(dir) || fs.readdirSync(dir).length === 0;
        //console.log(`Is Directory Empty (${dir}):`, isEmpty); // Debugging line
        return isEmpty;
    };

    const migrateSaveFiles = () => {
        const userSaveDir = getSaveDirectory();
        const gameSaveDir = getGameDirectory();

        if (isDirectoryEmpty(userSaveDir)) {
            if (fs.existsSync(gameSaveDir) && !isDirectoryEmpty(gameSaveDir)) {
                copySaveFiles(gameSaveDir, userSaveDir);
            }
        }
    };

    // Override save and load paths
    StorageManager.localFilePath = function(savefileId) {
        const saveDir = getSaveDirectory();
        const filename = 'file%1.rmmzsave'.replace('%1', savefileId);
        const fullPath = path.join(saveDir, filename);
       // console.log("Saving to:", fullPath); // Debugging line
        return fullPath;
    };
    
    StorageManager.fileDirectoryPath = function() {
        return getSaveDirectory();
    };

    // Block until migration is done
    const _Scene_Boot_create = Scene_Boot.prototype.create;
    Scene_Boot.prototype.create = function() {
        migrateSaveFiles(); // Synchronous migration
        _Scene_Boot_create.call(this);
    };

})();
