/*:
* https://notritter.itch.io/ritter-ultimate-event-spawner-rpg-maker-mz
*
* @target MZ
*
* @plugindesc Ritter Ultimate Event Spawner v1.9
* https://notritter.itch.io/
*
* @author Ritter
*
* @help
* --------------------------------------------------------------------------------
*   Ritter Ultimate Event Spawner Plugin Help File
* --------------------------------------------------------------------------------
*
* Ritter Ultimate Event Spawner gives you the power to easily
* Spawn events on the game map using templates from a separate map
* while reusing unspawned events to help prevent performance loss
* over time. No more erased/null event clutter bogging down your
* performance!
*
* This spawner should drastically improve performance when used
* properly, if you spawn too many events (ex: 1000+) without ever
* unspawning them you will likely see performance loss, this spawner
* is designed to reuse events that are unspawned so be sure to manage
* the number of active events you have on your map at once.
*
* --------------------------------------------------------------------------------
*   Script calls
* --------------------------------------------------------------------------------
*
* --------------------------------------------------------------------------------
*   Spawn an Event at x,y Location
* --------------------------------------------------------------------------------
*
* Ritter.spawnEvent(mapId, eventId, x, y, save)
*
* mapId = the Id of the map which holds the event you want to spawn.
* eventId = the Id of the event on the map specified in mapId.
* x = x coordinate to place the event on the game map.
* y = y coordinate to place the event on the game map.
* save = true to save event and respawn it when you re-enter map.
* save = false to use the event as a temporary event which goes away
*        on map change. You can leave out save from script call to not
* 		 save the event.
*
* -edit-
* -v1.2- Map/Event "Name" strings can be used for mapId and eventId. See examples below.
*
* Example script call:
* Ritter.spawnEvent(42, 15, 8, 16, true)
* 
* Would spawn event 15 from map 42 on the game map at 8,16 coordinates
* and save the event data to automatically be respawned upon re-entering
* the map.
*
* Example script call two:
* Ritter.spawnEvent(42, 15, 8, 16)
*
* Would spawn event 15 from map 42 on the game map at 8,16 coordinates
* as a temporary event that goes away on map change. 
*
* Example script call three:
* Ritter.spawnEvent("SpawnMap2", "Boo", 23, 42)
*
* Would spawn event named "Boo" from Spawn Map named "SpawnMap2" on the game map
* at 23, 42.
*
* Example script call four:
* Ritter.spawnEvent(5, "Boo", 23, 42, true);
*
* Would spawn event named "Boo" from Spawn Map 5 on the game map at 23, 42 and save it.
*
* --------------------------------------------------------------------------------
*   Spawn an Event On a Random Tile Within A Set of RegionIds
* --------------------------------------------------------------------------------
*
* Ritter.spawnEventRegion(mapId, eventId, regions, save)
*
* mapId = the Id of the map which holds the event you want to spawn.
* eventId = the Id of the event on the map specified in mapId.
* regions = regionId(s) to place the spawned event on randomly.
*           to use multiple regionIds to randomly pick from list them
*           like this [4,8,15,16,23,42]
* save = true to save event and respawn it when you re-enter map.
* save = false to use the event as a temporary event which goes away
*        on map change.
*
* Example script call:
* Ritter.spawnEventRegion(4, 8, 15, true)
*
* Would spawn event 8 from map 4 on a random tile marked with regionId 15
* and save the events data to be automatically respawned upon re-entering
* the map.
*
* Example script call with array of regions:
* Ritter.spawnEventRegion(4, 8, [15,16,23,42])
*
* Would spawn event 8 from map 4 on a random tile marked with any of the 
* regionIds 15,16,23,42 as a temporary event that isn't saved.
*
* --------------------------------------------------------------------------------
*   Transform an Event on the GameMap into an Event from a SpawnMap
* --------------------------------------------------------------------------------
*
* Ritter.transformEvent(eventId, mapId, spawnId)
*
* eventId = id of the event on game map to transform.
* mapId = spawn map id that has template event.
* spawnId = id of event on spawn map to use as template.
*
* Example Script call:
*
* Ritter.transformEvent(1010, 4, 8)
*
* Would transform event 1010 into event 8 from map 4.
*
* --------------------------------------------------------------------------------
*   Unspawn an Event from the GameMap
* --------------------------------------------------------------------------------
*
* Ritter.unspawnEvent(eventId, removeSave)
*
* eventId = the Id of the event on the map to unspawn.
* removeSave = true to remove the event from saved events.
* removeSave = false to keep the event on save list to respawn up re-entering
* 			   the map. Leave removeSave empty to keep saved event.
*
* Example script call:
* Ritter.unspawnEvent(1000, true)
*
* Would unspawn event 1000 from the game map and delete the saved event data.
*
* Example script call two:
* Ritter.unspawnEvent(1000)
*
* Would unspawn event 1000 from the game map.
*
* --------------------------------------------------------------------------------
*   Unspawn an Event from the GameMap using x,y Coordinates
* --------------------------------------------------------------------------------
*
* Ritter.unspawnEvent(x, y, removeSave);
*
* Unspawns a spawned event if found on tile x, y.
*
* Example script call:
*
* Ritter.unspawnEvent(7, 8);
*
* Would unspawn a spawned event found on tile located at 7, 8.
* 
* Ritter.unspawnEvent(7, 8, true);
*
* Would unspawn a spawned event found on tile located at 7, 8 and remove its save.
*
* --------------------------------------------------------------------------------
*   Unspawn ALL Spawned Events On the GameMap
* --------------------------------------------------------------------------------
* 
* Ritter.unspawnAll(removeSave)
*
* removeSave = true to delete saved event data.
* 			   leave empty to retain saved event data.
*
* Unspawns all spawned events on the map.
*
* --------------------------------------------------------------------------------
*   Restore an Unspawned Event - Might Delete
* --------------------------------------------------------------------------------
*
* Ritter.restoreUnspawnedEvent(eventId)
*
* Restores an unspawned saved event using its eventId to the game map.
*
* Ritter.restoreUnspawnedEvent("name")
*
* Restores an unspawned saved event using its name to the game map.
*
* Ritter.restoreUnspawnedEvent(x, y)
*
* Restores an unspawned event that was last unspawned on tile x, y.
*
* --------------------------------------------------------------------------------
*   Find Last Spawned EventId on the GameMap
* --------------------------------------------------------------------------------
*
* $gameMap._lastSpawnEventId
*
* Returns the Id of the most recently spawned event.
*
* --------------------------------------------------------------------------------
*   Boundary System Plugin Section
* --------------------------------------------------------------------------------
*
* Boundary System Script calls:  | Separate plugin required!
*                                | Requires Ritter_BoundarySystem.js
*                                | Place Ritter_BoundarySystem.js above
*                                | this plugin in plugin manager.
*
* When Using Script Calls:
* This is an advanced plugin, be sure to read this section carefully to
* fully understand how to use this powerful extension of the event spawner!
* 
* Using Plugin Commands + Auto Boundaries is best but script calls will remain
* for those already using them in their games.
*
* When Using Plugin Commands:
* This plugin is much easier to use with the automated boundary system.
* The Script Call help below may be useful if the help text in the plugin command 
* window doesn't suffice.
*
* For this plugin it's best to understand the difference between spawning on a
* boundary and spawning in a boundary. Spawning On a boundary means it will spawn
* an event on a valid random tile on the edge of the boundary. Spawning In a
* boundary means it will spawn an event on a valid random tile within the confines
* of a boundary.
* 
* Spawning On = ◻ vs Spawning In = ◼
* An Outline Of A Square vs A Filled-In Square.
*
* v1.5 Update:
*         MZ Plugin Commands and Auto Mode Added to Simplify Handling Boundaries.
*
* --------------------------------------------------------------------------------
*   Create a Spawner Boundary
* --------------------------------------------------------------------------------
*
* Ritter.Boundary.createSpawnerBoundary(width, height, thickness, name, eventId, expandBy, maxEvents, centerx, centery)
*
* width = total width of boundary.
*
* height = total height of boundary.
*
* thickness = the number of tiles thick the boundary is.
*
* name = the "name" of your boundary.
*
* eventId = 0 for game player, the id of any event for a boundary on an event, or -1 if using centerx, centery.
*
* expandBy = the number of tiles the boundary expands by in the direction the player is moving,
*			 this allows for increased probability of events spawning in front of the player as
*			 opposed to spawning on sides and back equally.
*
* maxEvents = the maximum number of events a boundary is allowed to spawn, this prevents the boundary
*			  from spawning an unlimited number of events and creating event clutter.
*
* centerx = leave these out if you're creating a boundary on player or event. set to x location if creating
*			boundary on x,y coordinates on map.
*
* centery = leave these out if you're creating a boundary on player or event. set to y location if creating
*			boundary on x,y coordinates on map.
*
* Example Script calls:
*
* Ritter.Boundary.createSpawnerBoundary(10, 10, 1, "spawn", 0, 2, 30)
*
* This would create a boundary of 10 tiles width and height around the game player with 1 thickness which is
* named "spawn", the boundary will expand by 2 tiles in the direction the player is moving and have a maximum
* of 30 events it can spawn.
*
* Ritter.Boundary.createSpawnerBoundary(13, 13, 1, "unspawn", 0, 0, 0)
*
* This would create a boundary of 13 tiles width and height around the game player with 1 thickness which is
* named "unspawn", the boundary will not expand in the direction the player is moving and will not be able
* to spawn any events.
*
* Ritter.Boundary.createSpawnerBoundary(4, 4, 2, "eventBoundary", 50, 0, 5)
*
* This would create a boundary of 4 tiles width and height around event 50 with 2 thickness which is named
* "eventBoundary", the boundary will not expand and is limited to spawning 5 events maximum at a time.
*
* Ritter.Boundary.createBoundary(5, 5, 1, "xyBoundary", -1, 0, 20, 50, 50)
*
* This would create a boundary of 5 tiles width and height around coordinates 50,50 on the game map which is
* named "xyBoundary", the boundary will not expand and is limited to spawning 20 events at a time.
*
* --------------------------------------------------------------------------------
*   Spawn an Event On the Edge of a Boundary
* --------------------------------------------------------------------------------
*
* Ritter.spawnEventOnBoundary(mapId, eventId, regions, boundaryName)
*
* mapId = The mapId of the spawn map which holds your template/source event to be copied to game map.
*
* eventId = The eventId of the template/source event to be copied to the game map.
*
* regions = The region Ids to use for spawning events on. List as an array to use multiple regionIds. [1, 2, 3]
*
* boundaryName = the "name" you gave your boundary that you wish to spawn on.
*
* --------------------------------------------------------------------------------
*   Spawn an Event Within a Boundary
* --------------------------------------------------------------------------------
*
* Ritter.spawnEventInBoundary(mapId, eventId, regions, boundaryName)
*
* mapId = The mapId of the spawn map which holds your template/source event to be copied to game map.
*
* eventId = The eventId of the template/source event to be copied to the game map.
*
* regions = The region Ids to use for spawning events on. List as an array to use multiple regionIds. [1, 2, 3]
*
* boundaryName = the "name" you gave your boundary that you wish to spawn on.
* 
* --------------------------------------------------------------------------------
*   Unspawn All Boundary Events On the Edge of a Boundary
* --------------------------------------------------------------------------------
*
* Ritter.unspawnEventOnBoundary(boundaryName)
*
* boundaryName = the "name" you assigned to the boundary you wish to use for unspawnEventOnBoundary.
*
*
* Example Script Call using boundaries created above:
*
* Ritter.unspawnEventOnBoundary("unspawn")
*
* This would unspawn any events located on the boundary "unspawn".
*
* --------------------------------------------------------------------------------
*   Unspawn All Boundary Events Within a Boundary
* --------------------------------------------------------------------------------
*
* Ritter.unspawnEventOnBoundary(boundaryName)
*
* boundaryName = the "name" you assigned to the boundary you wish to use for unspawnEventOnBoundary.
*
*
* Example Script Call using boundaries created above:
*
* Ritter.unspawnEventOnBoundary("unspawn")
*
* This would unspawn any events located on the boundary "unspawn".
*
* --------------------------------------------------------------------------------
*   Fill Every Tile Along the Edge of a Boundary with events.
* --------------------------------------------------------------------------------
*
* Ritter.spawnFillOnBoundary(mapId, eventId, regions, boundaryName);
*
* mapId = The mapId of the spawn map which holds your template/source event to be copied to game map.
*
* eventId = The eventId of the template/source event to be copied to the game map.
*
* regions = The region Ids to use for spawning events on. List as an array to use multiple regionIds. [1, 2, 3]
*
* boundaryName = the "name" you gave your boundary that you wish to spawn on.
*
* --------------------------------------------------------------------------------
*   Fill Every Tile Within a Boundary With Events.
* --------------------------------------------------------------------------------
*
* Ritter.spawnFillInBoundary(mapId, eventId, regions, boundaryName);
*
* mapId = The mapId of the spawn map which holds your template/source event to be copied to game map.
*
* eventId = The eventId of the template/source event to be copied to the game map.
*
* regions = The region Ids to use for spawning events on. List as an array to use multiple regionIds. [1, 2, 3]
*
* boundaryName = the "name" you gave your boundary that you wish to spawn on.
*
* --------------------------------------------------------------------------------
*   Suggested common event setup for spawning / unspawning on/in a boundary.
*   -  - (Optional!) New Update makes this optional, Auto Mode Available.
* --------------------------------------------------------------------------------
*
* Create a common event that runs parallel and tie it to a switch, so that when you enable the switch
* the spawn event functions get called.
*
* First common event:
*
* Script: Ritter.spawnEventOnBoundary(mapId, eventId, regions, boundaryName)
* Wait: 30-60 frames. (Suggested to use 30-40 frame wait minimum)
*
* Second common event:
*
* Ritter.unspawnEventOnBoundary(boundaryName)
* Wait: 10 frames. (Suggested 10 frame wait per thickness of unspawn boundary)
*				   (10 frames * thickness)
*
* --------------------------------------------------------------------------------
*   v1.5 Boundary System Spawner Update Additions
* --------------------------------------------------------------------------------
* Version 1.5 of Ritter Ultimate Event Spawner brings about big changes for
* the boundary system. The biggest change would be Auto Boundaries! Below we'll
* go through every new feature and explain how and when to use them.
*
* Setting Up An Auto Boundary:
*
* Use the plugin command 'CreateBoundary' and fill out the properties.
* You can find information about the properties in the script call section above
* for CreateBoundary.
*
* Immediately following that plugin command use the plugin command 'AddAutoHandler'.
* This plugin command tells the spawner that the boundary is to be ran on auto.
*
* 'AddAutoHandler' Properties:
*
* Name = The Name of the Boundary you just created.
* Spawn Map Id = The ID # of a Spawn Map you want the boundary to use.
* Map Ids = A List of Map Ids the Boundary will run on.
* Boundary Type = The Type of Boundary you want. SpawnOn, SpawnIn, FillOn, FillIn, 
*                 UnspawnOn, UnspawnIn.
* Wait Time = The number of frames to wait between boundary actions. (Spawn/Unspawn)
* Enabled/Disabled By Default = Whether the boundary enables itself upon Creation.
*                               Enabled boundaries will only run when on proper map(s).
*                               Disabled boundaries will not run at all until activated.
* Unspawn Settings = These are only required for Unspawn Boundaries.
* Enable Unspawn Boundary = This tells the spawner to treat as unspawn boundary. true/false
* Target Boundaries = A list of Boundary Names this Unspawn Boundary may unspawn events from.
*                     Boundary list must contain events parent boundary to unspawn an event.
*
* ---- Auto Boundary Spawn Event Setup ----
* 
* Great! Now we have our Auto Boundaries set up. Now lets look at how to set up
* Spawn Events to work with Auto Boundaries.
*
* Go to your Spawn Map of Choice and Pick an Event.
* On the First Page add the Plugin Command 'BoundaryEventSetup'
* (This plugin command is parsed by the plugin and does not run any code itself)
* 
* 'BoundaryEventSetup' Properties:
*
* Boundary List = A List of Boundaries this Event may Spawn on.
* Map List = A List of Map Ids this Event may Spawn on.
* RegionId List = A List of Region Ids this Event may Spawn on.
* Enabled By Default? = Whether this Event is Enabled/Disabled by Default.
* Saved Event = Whether the event data is saved and restored upon respawn.
*               Only eventId changes for Saved Boundary Events all other event data remains.
*               You may use (practically) as many saved events as you wish, one per tile.
*
* ---- Test It Out ----
*
* Assuming you set the Auto Handler(s) to Enabled By Default just load up a valid
* map and run to valid regions where events can spawn. 
* 
* --------------------------------------------------------------------------------
*   Terms of Use
* --------------------------------------------------------------------------------
*
* Can be used in Commercial and Non-Commercial games if purchased.
* Personal edits to code are allowed for use in your game.
* Don't claim credit for creating this plugin.
* Don't post this plugin elsewhere, modified or otherwise.
* Don't remove my name from Author.
* I am not liable for any problems you may encounter while
* using this plugin, though I may, or may not, be available
* for support on RPG Maker Forums or my itch.io page for this plugin.
* I will not offer support for edits to the code.
* Please credit me in your games credits as: Craig "Ritter" B.
*
*
* --------------------------------------------------------------------------------
*   Version log
* --------------------------------------------------------------------------------
*
* 1.0: Released Plugin.
* 1.1: Discovered, Isolated, and Fixed a rare crash related to unspawning
* 	   an unspawned event and then changing scenes.
* 1.2: Added spawning events by "SpawnMapName" & "EventName".
*      Added unspawning events by x, y location.
*      Added ability to easily edit boundary properties.
*      Added new boundary script calls for filling boundaries with events.
*      Added unspawnEventInBoundary script call.
* 1.5: Added Plugin Commands for Event Spawner 
*      Added Plugin Commands for Boundary System.
*      Added Automated Boundaries.
*      Added Saved Boundary Events.
*      Added Developer Mode Warning Alerts. (Work in progress)
*      Added Info Text Throughout Plugin Code.
*      Fixed an issue with recycled event sprites not being removed completely.
* 1.6: Fixed a Bug with Boundary Saved Events.
*      Made Load Spawn Map code cleaner and more efficient.
*      Moved all SpawnMap Data into one object. $MapData[mapId].
* 1.7: Now compatible with RPG Maker MZ Corescript v1.7.0
* 1.8: Made some optimization changes to Boundary Systems Auto Boundaries.
* 1.9: Fixed a crash related to saved events not retaining data under a specific circumstance.
*
* --------------------------------------------------------------------------------
*   Plugin Parameters
* --------------------------------------------------------------------------------
*
* @param Spawn Events Starting Id
* @type number
* @default 1000
* @desc The ID Number to use as the first spawn event ID.
*
* @param Preload Spawn Maps
* @type string
* @desc Optional- list spawn maps to preload. Separate with spaces.
* Example: 4 8 15 16 23 42
*
* @param Developer Mode
* @type boolean
* @default false
* @desc Shows alerts to developer when mistakes or errors could cause the
* Plugin to function improperly. Suggests how to fix the problem.
* 
* @param Developer Mode Alert Settings
* @type struct<DebugAlerts>
* @default {"Spawn1":"true","Spawn2":"true"}
* @desc Enable/Disable Various Developer Mode Alerts.
* Work In Progress.. Incomplete.
*
* --------------------------------------------------------------------------------
*   Plugin Commands
* --------------------------------------------------------------------------------
*
* @command SpawnEvent
* @desc Spawn an event on the game map.
* 
* @arg mapId
* @text Map ID
* @desc The Id Number or Name of the map which holds the event you want to spawn.
* @type text
* 
* @arg eventId
* @text Event ID
* @desc The Id Number or Name of the event to spawn.
* @type text
*
* @arg X
* @text X Location
* @desc X coordinate to place the event on the game map.
* @type text
*
* @arg Y
* @text Y Location
* @desc Y coordinate to place the event on the game map.
* @type text
*
* @arg save
* @text Saved Event?
* @desc true for permanent event. False for temporary event.
* @type boolean
* @default false
*
* @command SpawnEventRegion
* @desc Spawn event on a random tile within a region.
*
* @arg mapId
* @text Spawn Map Id
* @desc Id of Spawn Map
* @default 0
* @type number
*
* @arg eventId
* @text Spawn Map Event Id
* @desc Event Id from Spawn map to spawn on Game Map.
* @default 0
* @type number
* 
* @arg regions
* @text Region Id(s)
* @desc Valid Region Ids to spawn on. To randomly pick from list of RegionIds just separate with comma: 1,2,3
* @default 0
* @type text
*
* @arg save
* @text Saved Event?
* @desc true for permanent event. False for temporary event.
* @type boolean
* @default false
*
* @command TransformEvent
* @desc Transform an event on the game map into another event from a spawn map.
*
* @arg eventId
* @text Target Event Id
* @desc Id of Event on Game Map to Transform.
* Use 'this' to transform this event on the game map.
* @default 0
* @type text
*
* @arg mapId
* @text Spawn Map Id
* @desc Id of Spawn Map
* @default 0
* @type number
*
* @arg spawnId
* @text Spawn Map Event Id
* @desc Id of Event on Spawn Map to use as template.
* @default 0
* @type number
*
* @command UnspawnEvent
* @desc Unspawn an Event.
*
* @arg eventId
* @desc EventId of an Event on the Game Map to unspawn.
* Use 'this' to unspawn this event.
* @default 0
* @type text
* 
* @arg removeSave
* @desc Remove Saved Event Data?
* @default false
* @type boolean
*
* @command UnspawnEventXY
* @desc Unspawn an Event at X, Y Coordinates.
*
* @arg X
* @text X Location
* @desc X location of Event to unspawn.
* @default 0
* @type text
*
* @arg Y
* @text Y Location
* @desc Y location of Event to unspawn.
* @default 0
* @type text
* 
* @arg removeSave
* @text Remove Saved Event Data?
* @desc Remove Saved Event Data?
* @default false
* @type boolean
*
* @command UnspawnAll
* @desc Unspawn all events.
* 
* @arg removeSaves
* @text Remove Saved Event Data?
* @desc Removes all saved event data for this game map.
* Does not remove saved data for Boundary Events.
* @default false
* @type boolean
*
* @command CreateBoundary
* @desc Create a Spawn Boundary
* 
* @arg Name
* @text Boundary Name
* @desc The name of your boundary. Works as an Id for spawning.
* @type text
* @default BoundaryName
*
* @arg Width
* @text Boundary Width
* @desc The number of tiles wide the boundary is.
* @type number
*
* @arg Height
* @text Boundary Height
* @desc The number of tiles high the boundary is.
* @type number
*
* @arg Thickness
* @text Boundary Thickness
* @desc The number of tiles thick the boundary is.
* @type number
* @default 1
*
* @arg EventId
* @text Event Id
* @desc Id of target to create boundary around. Use 0 for player, -1 for an x,y location, or eventId for events.
* @type number
* @default 0
* @min -1
*
* @arg ExpandBy
* @text Expand By
* @desc The number of tiles the boundary expands its thickness by in the direction of the players movement.
* @type number
* @default 1
*
* @arg MaxEvents
* @text Max Events
* @desc Max Events this Boundary can spawn.
* @type number
* @default 30
*
* @arg CenterX
* @text Center X
* @desc The X location of the center of the Boundary. Leave undefined unless creating boundary on x,y.
* @type number
* @default undefined
*
* @arg CenterY
* @text Center Y
* @desc The Y location of the center of the Boundary. Leave undefined unless creating boundary on x,y.
* @type number
* @default undefined
* 
* @command AddAutoHandler
* @desc Add an Auto Handler to a boundary allowing for the boundary to run automatically.
* 
* @arg Name
* @text Boundary Name
* @desc The name of the boundary which you wish to run automatically.
* @type text
* @default BoundaryName
*
* @arg SpawnMap
* @text Spawn Map Id
* @desc The Spawn Map Id to be used for this boundary.
* @type number
* @default 1
*
* @arg Maps
* @text Map Ids
* @desc List of Map Ids this Boundary can run on.
* @type number[]
* @default []
*
* @arg Type
* @text Boundary Type
* @desc Type of Boundary.
* @type select
* @option Spawn Event On The Boundary Edge
* @value SpawnOn
* @option Spawn Event Within The Boundary
* @value SpawnIn
* @option Fill Every Tile On The Boundary Edge
* @value FillOn
* @option Fill Every Tile Within The Boundary
* @value FillIn
* @option Unspawn Event On The Boundary Edge
* @value UnspawnOn
* @option Unspawn Event Within The Boundary
* @value UnspawnIn
* @default SpawnOn
*
* @arg Wait
* @text Wait Time
* @desc Wait Time (In Frames) Between Boundary Actions.
* @type number
* @default 10
*
* @arg Enabled
* @text Boundary Enabled/Disabled
* @desc Boundary Enabled Or Disabled By Default?
* @type boolean
* @default true
*
* @arg UnspawnSettings
* @text Unspawn Settings
* @desc Settings only used for Unspawn Boundaries
* @type struct<UnspawnSetting>
* @default {"Unspawn":"false","Boundaries":"[]"}
* 
* @command EditBoundary
* @desc Edit the properties of a Boundary.
*
* @arg Name
* @text Boundary Name
* @desc The Boundary you wish to edit.
* @type text
*
* @arg Width
* @text Width
* @desc Width of Boundary.
* @type number
* @default undefined
*
* @arg Height
* @text Height
* @desc Height of Boundary.
* @type number
* @default undefined
*
* @arg Thickness
* @text Thickness
* @desc Thickness of Boundary.
* @type number
* @default undefined
*
* @arg ExpandBy
* @text Expand By
* @desc Number of Tiles a Boundary Expands by in the direction the player is moving.
* @type number
* @default undefined
*
* @arg X
* @text Center X
* @desc The X coordinate of the center of a Boundary.
* @type number
* @default undefined
* 
* @arg Y
* @text Center Y
* @desc The Y coordinate of the center of a Boundary.
* @type number
* @default undefined
*
* @arg MaxEvents
* @text Max Events
* @desc The Maximum number of events the boundary may spawn.
* @type number
* @default undefined
*
* @arg WaitTime
* @text Wait Time
* @desc The Wait Timer in frames for an Automatic Boundary.
* @type number
* @default undefined
*
* @command BoundaryEventSetup
* @desc This Plugin Command must be placed on the First Page of all Auto Boundary Events.
*
* @arg BoundaryList
* @text Boundary List
* @desc List of Boundary Names this event may spawn on.
* @type text[]
* @default []
* @min 0
*
* @arg MapList
* @text Map List
* @desc List of MapIds this event may spawn on.
* @type text[]
* @default []
* @min 1
*
* @arg RegionsList
* @text RegionId List
* @desc List of all regionIds this event may spawn on.
* @type number[]
* @default []
* @min 0
*
* @arg Enabled
* @text Enabled by Default?
* @desc Whether spawning this Event is Enabled or Disabled by default. Can be toggled on/off with a plugin command.
* @type boolean
* @default true
*
* @arg Save
* @text Saved Event?
* @desc Whether the event is saved or not. Will respawn if tile passes through a valid spawn boundary.
* @type boolean
* @default false
*
* @command InitializeBoundary
* @desc Initialize a Boundary. Spawns any saved boundary events found within a boundary.
* Suggested: Place/Include in an auto-run event that loads on map changes and erases itself.
*
* @arg Name
* @text Boundary Name
* @desc Name of Boundary to Initialize Events.
* @type text
*
* @command ActivateBoundary
* @desc Turns ON a specified Boundary.
*
* @arg Name
* @text Boundary Name
* @desc Name of Boundary to Activate.
* @type text
*
* @command DeactivateBoundary
* @desc Turns OFF a specified Boundary.
*
* @arg Name
* @text Boundary Name
* @desc Name of Boundary to Deactivate.
* @type text
*
* @arg UnspawnAll
* @text Unspawn All Boundary Events?
* @desc Unspawns All Events Spawned By This Boundary.
* @type boolean
* @default false
*
* @command EnableEvent
* @desc Enables a Spawn Event so it can Spawn on Boundaries.
* 
* @arg SpawnMap
* @text Spawn Map Id
* @desc Map Id of Spawn Map holding Event to Enable.
* @type number
* 
* @arg EventId
* @text Event Id
* @desc Event Id on the Spawn Map to Enable.
* @type number
*
* @command DisableEvent
* @desc Disables a Spawn Event so it won't Spawn on Boundaries.
* 
* @arg SpawnMap
* @text Spawn Map Id
* @desc Map Id of Spawn Map holding Event to Disable.
* @type number
* 
* @arg EventId
* @text Event Id
* @desc Event Id on Spawn Map to Disable.
* @type number
*
*/
/*~struct~DebugAlerts:
* @param Spawn1
* @text Spawn Error 1
* @type boolean
* @default true
* @desc Enable/Disable 'Unable to locate event by name' Warning.
*
* @param Spawn2
* @text Spawn Error 2
* @type boolean
* @default true
* @desc Enable/Disable 'Unable to locate Spawn Map by name' Warning.
*/
/*~struct~UnspawnSetting:
* @param Unspawn
* @text Enable Unspawn Boundary?
* @type boolean
* @default false
* @desc Must be enabled for unspawn boundary to work. Disable for Spawn Boundaries.
*
* @param Boundaries
* @text Target Boundaries
* @type text[]
* @default []
* @desc List of Boundaries this Unspawn Boundary may Unspawn events from.
*/

var Imported = Imported || {};
Imported.Ritter_EventSpawner = true;

var Ritter = Ritter || {};

/**
 * --------------------------------------------------------------------------------
 *   Plugin Parameters Code
 * --------------------------------------------------------------------------------
 */

Ritter.Params = Ritter.Params || {};
Ritter.Boundary = Ritter.Boundary || {};
Ritter._SpawnMaps = Ritter._SpawnMaps || [];
Ritter._mapRegionData = [];
$gameMap = $gameMap || {};
$gameMap._savedEvents = [];
$gameMap._spawnedList = [];
$gameMap._unspawnList = [];
$MapData = [];
if (Imported.Ritter_BoundarySystem) {
	$dataBoundary = {};
}

var parameters = PluginManager.parameters('Ritter_EventSpawner');

Ritter.Params.FirstId = Number(parameters["Spawn Events Starting Id"]); 

Ritter.Params.preloadMaps = String(parameters["Preload Spawn Maps"]);
if (Ritter.Params.preloadMaps) {
	Ritter.Params.preloadMaps = Ritter.Params.preloadMaps.split(' ');
	for (var i = 0; i < Ritter.Params.preloadMaps.length; i++) {
		Ritter.Params.preloadMaps[i] = Number(Ritter.Params.preloadMaps[i]);
	}
}
Ritter.Params.devMode = eval(parameters["Developer Mode"]);
Ritter.Params.devAlerts = JSON.parse(parameters["Developer Mode Alert Settings"]);

/**
 * --------------------------------------------------------------------------------
 *   Map Region Data Code
 * --------------------------------------------------------------------------------
 */
 Ritter_Game_Map_Setup_Spawn = Game_Map.prototype.setup;
 Game_Map.prototype.setup = function(mapId) {
	 Ritter_Game_Map_Setup_Spawn.call(this,mapId);
	 Ritter.getRegionData();
 };
 
 Ritter.getRegionData = function(refresh = false) {
	 mapId = $gameMap._mapId;
	 if (refresh === true) delete Ritter._mapRegionData[mapId];
	 if (!Ritter._mapRegionData[mapId]) Ritter._mapRegionData[mapId] = [];
	 for (let x = 0, w = $gameMap.width(); x < w; x++) {
		 for (let y = 0, h = $gameMap.height(); y < h; y++) {
			if (!Ritter._mapRegionData[mapId][$gameMap.regionId(x,y)]) {
				Ritter._mapRegionData[mapId][$gameMap.regionId(x,y)] = [];
			}
			Ritter._mapRegionData[mapId][$gameMap.regionId(x,y)].push([x,y]);
		}
	}
 };

/**
 * --------------------------------------------------------------------------------
 *   Plugin Commands Code
 * --------------------------------------------------------------------------------
 */

/**
 * Modifies Game Interpreter Plugin Command to set 'this' eventId when unspawning
 * or Transforming an event if value set is Not a Number. Any string will work
 * Suggested: 'this'
 */
Ritter_Game_Interpreter_command357 = Game_Interpreter.prototype.command357
Game_Interpreter.prototype.command357 = function(params) {
	if (params[0] === "Ritter_EventSpawner" && params[1] === "UnspawnEvent" && isNaN(params[3].eventId)) {
		params[3].eventId = this._eventId;
	}
	if (params[0] === "Ritter_EventSpawner" && params[1] === "TransformEvent" && isNaN(params[3].eventId)) {
		params[3].eventId = this._eventId;
	}
	return Ritter_Game_Interpreter_command357.call(this,params);
};

PluginManager.registerCommand('Ritter_EventSpawner', 'SpawnEvent', (args) => {
	if (!isNaN(args.mapId)) args.mapId = parseInt(args.mapId);
	if (!isNaN(args.eventId)) args.eventId = parseInt(args.eventId);
	const mapId = args.mapId;
	const eventId = args.eventId;
	const x = eval(args.X);
	const y = eval(args.Y);
	const save = eval(args.save);
	Ritter.spawnEvent(mapId, eventId, x, y, save);
});

PluginManager.registerCommand('Ritter_EventSpawner', 'UnspawnAll', (args) => {
	Ritter.unspawnAll(eval(args.removeSaves));
});

PluginManager.registerCommand('Ritter_EventSpawner', 'UnspawnEvent', (args) => {
	const eventId = Number(args.eventId);
	const save = eval(args.removeSave);
	Ritter.unspawnEvent(eventId, save);
});

PluginManager.registerCommand('Ritter_EventSpawner', 'UnspawnEventXY', (args) => {
	const x = eval(args.X);
	const y = eval(args.Y);
	const save = eval(args.removeSave);
	Ritter.unspawnEvent(x, y, save);
});

PluginManager.registerCommand('Ritter_EventSpawner', 'TransformEvent', (args) => {
	Ritter.transformEvent(
		Number(args.eventId),
		Number(args.mapId),
		Number(args.spawnId));
});

PluginManager.registerCommand('Ritter_EventSpawner', 'SpawnEventRegion', (args) => {
	regions = args.regions.split(",");
	for (let i = 0; i < regions.length; i++) {
		regions[i] = Number(regions[i]);
	}
	Ritter.spawnEventRegion(
		Number(args.mapId),
		Number(args.eventId),
		regions,
		eval(args.save));
});

PluginManager.registerCommand('Ritter_EventSpawner', 'CreateBoundary', (args) => {
	Ritter.Boundary.createSpawnerBoundary(
		Number(args.Width),
		Number(args.Height),
		Number(args.Thickness),
		String(args.Name),
		Number(args.EventId),
		Number(args.ExpandBy),
		Number(args.MaxEvents),
		Number(args.CenterX),
		Number(args.CenterY));
});

PluginManager.registerCommand('Ritter_EventSpawner', 'AddAutoHandler', (args) => {
	maps = args.Maps;
	for (let i = 0; i < maps.length; i++) {
		maps[i] = Number(maps[i]);
	}
	var Boundaries = [];
	unspawnSettings = JSON.parse(args.UnspawnSettings);

	if (unspawnSettings.Unspawn) Boundaries = JSON.parse(unspawnSettings.Boundaries);
	Ritter.Boundary.addAutoHandler(
		args.Name,
		Number(args.SpawnMap),
		maps,
		String(args.Type),
		Number(args.Wait),
		eval(args.Enabled),
		Array(Boundaries));
});

PluginManager.registerCommand('Ritter_EventSpawner', 'EditBoundary', (args) => {
	const boundaryName = args.Name;
	if (args.Width != "undefined") Ritter.Boundary.editSpawnerBoundary(args.Name, "width", args.Width);
	if (args.Height != "undefined") Ritter.Boundary.editSpawnerBoundary(args.Name, "height", args.Height);
	if (args.Thickness != "undefined") Ritter.Boundary.editSpawnerBoundary(args.Name, "thickness", args.Thickness);
	if (args.ExpandBy != "undefined") Ritter.Boundary.editSpawnerBoundary(args.Name, "expandBy", args.ExpandBy);
	if (args.X != "undefined") Ritter.Boundary.editSpawnerBoundary(args.Name, "centerx", args.X);
	if (args.Y != "undefined") Ritter.Boundary.editSpawnerBoundary(args.Name, "centery", args.Y);
	if (args.MaxEvents != "undefined") Ritter.Boundary.editSpawnerBoundary(args.Name, "maxEvents", args.MaxEvents);
	if (args.WaitTime != "undefined") Ritter.Boundary.editSpawnerBoundary(args.Name, "waittime", args.WaitTime);
});

PluginManager.registerCommand('Ritter_EventSpawner', 'InitializeBoundary', (args) => {
	const boundaryName = args.Name;
	Ritter.Boundary.initBoundaryEvents(boundaryName);
});

PluginManager.registerCommand('Ritter_EventSpawner', 'ActivateBoundary', (args) => {
	const boundaryName = args.Name;
	Ritter.Boundary.activate(boundaryName);
});

PluginManager.registerCommand('Ritter_EventSpawner', 'DeactivateBoundary', (args) => {
	const boundaryName = args.Name;
	const unspawnAll = eval(args.UnspawnAll);
	Ritter.Boundary.deactivate(boundaryName, unspawnAll);
});

PluginManager.registerCommand('Ritter_EventSpawner', 'EnableEvent', (args) => {
	Ritter.Boundary.enableEvent(args.SpawnMap, args.EventId);
});

PluginManager.registerCommand('Ritter_EventSpawner', 'DisableEvent', (args) => {
	Ritter.Boundary.disableEvent(args.SpawnMap, args.EventId);
});

(function() {
	
/**
 * --------------------------------------------------------------------------------
 *   Spawn Map Data Preloading
 * --------------------------------------------------------------------------------
 */
	
DataManager.preloadMaps = function() { 
	if (!Ritter.Params.preloadMaps) return;
	var mapIds = Ritter.Params.preloadMaps;
	for (var i = 0; i < Ritter.Params.preloadMaps.length; i++) {
		DataManager.loadSpawnMap(mapIds[i]);
	}
};

/**
 * Loads Map Data for Spawner.
 * Can also spawn events on map load.
 * This allows for you to spawn an event from any map
 * without preloading.
 * Just enter mapId and eventId into spawnEvent to spawn
 * any event from any map without preloading.
 * 
 * @param {number} mapId - The Map Id # or "Map Name".
 * @param {number} eventId - The Event Id # or "Event Name".
 * @param {number} x - The X Location to Spawn an Event.
 * @param {number} y - The Y Location to Spawn an Event.
 * @param {boolean} save - Spawn as Saved Event?
 */
DataManager.loadSpawnMap = function(mapId, eventId, x, y, save) {
	var map = new XMLHttpRequest();
	var url = 'data/Map%1.json'.format(mapId.padZero(3));
	map.open("GET", url);
	map.overrideMimeType("application/json");
	map.onload = function() {
		Ritter._SpawnMaps.push(mapId);
		const mapData = JSON.parse(map.responseText);
		$MapData[mapId].events = mapData.events;
		$MapData[mapId].events.forEach(event => {
			if (event !== null) DataManager.extractMetadata(event);
		})
		if (Imported.Ritter_BoundarySystem) Ritter.Boundary.extractBoundaryData(mapData);
		if (typeof eventId === "string") eventId = Ritter._getEventIdFromString(mapId, eventId);
		if (eventId > 0) $gameMap.spawnEvent(mapId, eventId, x, y, save);
	};
	map.onerror = function() { throw new Error('Failed to load: ' + url); };
	map.send();
};

/**
 * --------------------------------------------------------------------------------
 *   Event Spawner Code
 * --------------------------------------------------------------------------------
 */

/**
 * Spawn an Event at x,y Location
 * 
 * @param {number} mapId - MapId of a SpawnMap
 * @param {number} eventId - EventId or "Name" from a SpawnMap
 * @param {number} x - X Location to Spawn an Event
 * @param {number} y - Y Location to Spawn an Event
 * @param {boolean} save - Saved Event Data?
 */
Ritter.spawnEvent = function(mapId, eventId, x, y, save) {
	if (!(SceneManager._scene instanceof Scene_Map)) return;
	if (!Ritter.canSpawnOn(x, y)) return;
	if (!Ritter._SpawnMaps.contains(mapId)) {
		if (typeof mapId === "string") mapId = Ritter._getMapIdFromString(mapId);
		DataManager.loadSpawnMap(mapId, eventId, x, y, save);
	} else { 
		$gameMap.spawnEvent(mapId, eventId, x, y, save); 
	}
};

/**
 * Spawn an Event on a Random Tile in a Region.
 * 
 * @param {number} mapId - MapId of a SpawnMap
 * @param {number} eventId - EventId from a SpawnMap
 * @param {array} regions - Array of RegionIds to Spawn On
 * @param {boolean} save - Saved Event Data?
 * @returns 
 */
Ritter.spawnEventRegion = function(mapId, eventId, regions, save) {
	if (!(SceneManager._scene instanceof Scene_Map)) return;
	$gameMap.spawnEventRegion(mapId, eventId, regions, save);
};

/**
 * Transform an Event on the GameMap into an Event from a SpawnMap.
 * 
 * @param {number} eventId - EventId from the GameMap to Transform
 * @param {number} mapId - SpawnMap Id
 * @param {number} spawnId - Event Id from the SpawnMap
 */
Ritter.transformEvent = function(eventId, mapId, spawnId) {
	if (!(SceneManager._scene instanceof Scene_Map)) return;
	$gameMap.transformEvent(eventId, mapId, spawnId);
};

/**
 * Unspawn an Event.
 * 
 * @param {number} eventId - EventId from the GameMap to Unspawn
 * @param {boolean} removeSave - Remove Saved Event Data?
 * @param {boolean} arg - Becomes removeSave if x, y is used instead of eventId
 */
Ritter.unspawnEvent = function(eventId, removeSave, arg = false) {
	// Checking if removeSave is set to integer
	// if so we handle unspawning an event on an x,y location.
	if (Number.isInteger(removeSave)) {
		const x = eventId;
		const y = removeSave;
		eventId = $gameMap.eventIdXy(x, y);
		removeSave = arg;
	}

	$gameMap.unspawnEvent(eventId, removeSave);
};

/**
 * Unspawn All Spawned Events on the Game Map.
 * 
 * @param {boolean} removeSave - Remove Saved Event Data?
 */
Ritter.unspawnAll = function(removeSave = false) {
	$gameMap.unspawnAll(removeSave);
};

/**
 * Restore an Unspawned Event.
 * 
 * @param {number} eventId - EventId to Restore
 */
Ritter.restoreUnspawnedEvent = function(eventId) {
	$gameMap.restoreUnspawnedEvent(eventId);
};

const Ritter_Game_Event_initialize = Game_Event.prototype.initialize;
/**
 * Adding ._parentBoundary property to event initialize.
 */
Game_Event.prototype.initialize = function(mapId, eventId) {
	Ritter_Game_Event_initialize.call(this, mapId, eventId);
	this._parentBoundary = "";
};

/**
 * Handles Initialization and MetaDeta of New Events.
 * 
 * @param {object} event - GameMap Event Object
 */
Game_Event.prototype.newData = function(event) {
	this._eventData = event;
	this.initialize(this._mapId, this._eventId);
	DataManager.extractMetadata(this.event());
	DataManager.extractMetadata(event);
};

/**
 * Get an Events Id from a Spawn Map using its Name.
 * 
 * @param {number} mapId - SpawnMap Id
 * @param {string} string - A "String" containing an Event Name from a SpawnMap
 */
Ritter._getEventIdFromString = function(mapId, string) {
	let mapData = $MapData[mapId];
	let mId = [];
	mapData.events.forEach((event) => {
		if (!!event && event.name == string) mId.push(event.id);
	});
	if (!mId.length && Ritter.Params.devMode && Ritter.Params.devAlerts.Spawn1) {
		alert(
		"⚡ 𝗥𝗶𝘁𝘁𝗲𝗿_𝗘𝘃𝗲𝗻𝘁𝗦𝗽𝗮𝘄𝗻𝗲𝗿.𝗷𝘀 : 𝗘𝗩𝗘𝗡𝗧 𝗦𝗣𝗔𝗪𝗡𝗜𝗡𝗚 : 𝗪𝗔𝗥𝗡𝗜𝗡𝗚! ⚡\n" +
		"🡺 Unable to locate event by name 🡸\n\n" +
		"Could not find an Event named: '" + string + "'. \n" + 
		"Did you use the correct Event Name or Spawn Map?\n\n" +
		"Disable Spawn Warning 1 in Plugin Parameters to never see this again."
		);
		return;
	}
	return mId[0];
};

/**
 * Get a SpawnMap Id using its Name.
 * 
 * @param {string} string - A "String" containing a SpawnMap Name
 */
Ritter._getMapIdFromString = function(string) {
	let id = [];
	$dataMapInfos.forEach((map) => 
	{
		if (!!map && string == map.name) { id.push(map.id); }
	});
	if (!id.length && Ritter.Params.devMode && Ritter.Params.devAlerts.Spawn2) {
		alert(
		"⚡ 𝗥𝗶𝘁𝘁𝗲𝗿_𝗘𝘃𝗲𝗻𝘁𝗦𝗽𝗮𝘄𝗻𝗲𝗿.𝗷𝘀 : 𝗘𝗩𝗘𝗡𝗧 𝗦𝗣𝗔𝗪𝗡𝗜𝗡𝗚 : 𝗪𝗔𝗥𝗡𝗜𝗡𝗚! ⚡\n" +
		"🡺 Unable to locate Spawn Map by name 🡸\n\n" +
		"Could not find a Map named: '" + string + "'.\n" +
		"Did you use the correct Name?\n\n" +
		"Disable Spawn Warning 2 in Plugin Parameters to never see this again."
		);
		return;
	}
	return id[0];
};

/**
 * Spawn an Event at x,y Location
 * 
 * @param {number} mapId - MapId of a SpawnMap
 * @param {number} eventId - EventId or "Name" from a SpawnMap
 * @param {number} x - X Location to Spawn an Event
 * @param {number} y - Y Location to Spawn an Event
 * @param {boolean} save - Saved Event Data?
 */
Game_Map.prototype.spawnEvent = function(mapId, eventId, x, y, save) {
	if (!this._unspawnList) this._unspawnList = [];
	if (!this._spawnedList) this._spawnedList = [];
	if (!mapId) return;
	if (!eventId) return;
	if (typeof mapId === "string") mapId = Ritter._getMapIdFromString(mapId);
	if (typeof eventId === "string") eventId = Ritter._getEventIdFromString(mapId, eventId);

	var nextId = this.getFirstNullEvent();
	var mapData = $MapData[mapId];
	
	if (Ritter.Params.devMode && mapData == undefined) {
		alert(
		"⚡ 𝗥𝗶𝘁𝘁𝗲𝗿_𝗘𝘃𝗲𝗻𝘁𝗦𝗽𝗮𝘄𝗻𝗲𝗿.𝗷𝘀 : 𝗘𝗩𝗘𝗡𝗧 𝗦𝗣𝗔𝗪𝗡𝗜𝗡𝗚 : 𝗪𝗔𝗥𝗡𝗜𝗡𝗚! ⚡\n" +
		"🡺 Unable to locate Map Data for MapID: " + mapId + " 🡸\n\n" +
		"Could not find an Event to spawn with id: " + eventId + " \n" + 
		"On Spawn Map: " + mapId + " \n" + 
		"Did you use the correct Event Id or Spawn Map?"
		);
		return;
	}
	
	var event = mapData.events[eventId];
	
	if (Ritter.Params.devMode && mapData.events[eventId] == undefined) {
		alert(
		"⚡ 𝗥𝗶𝘁𝘁𝗲𝗿_𝗘𝘃𝗲𝗻𝘁𝗦𝗽𝗮𝘄𝗻𝗲𝗿.𝗷𝘀 : 𝗘𝗩𝗘𝗡𝗧 𝗦𝗣𝗔𝗪𝗡𝗜𝗡𝗚 : 𝗪𝗔𝗥𝗡𝗜𝗡𝗚! ⚡\n" +
		"🡺 Unable to locate spawnEvent with eventId " + eventId + " 🡸\n\n" +
		"Could not find an Event to spawn with id: " + eventId + " \n" + 
		"On Spawn Map: " + mapId + " \n" + 
		"Did you use the correct Event Id or Spawn Map?"
		);
		return;
	}
	
	var event = JsonEx.makeDeepCopy(event);
	event.x = x;
	event.y = y;

	
	
	if (this._unspawnList.length > 0) { 
		var spawnEvent = $gameMap._events[this._unspawnList[0]];
		event.id = this._unspawnList[0];
		$dataMap.events[this._unspawnList[0]] = event;
		spawnEvent.newData(event);
		this._spawnedList.push(event.id);
		this._unspawnList.splice(0, 1);
	} else {
		event.id = nextId;
		$dataMap.events[nextId] = event;
		var spawnEvent = new Game_Event(this._mapId, nextId);
		spawnEvent.newData(event);
		spawnEvent._eventId = nextId;
		this._events[nextId] = spawnEvent;
		this._spawnedList.push(nextId);
		
	}
	
	
	this._lastSpawnEventId = spawnEvent._eventId;
	spawnEvent._spawnMap = mapId;
	spawnEvent._spawnId = eventId;
	if (save) this.saveEvent(spawnEvent._eventId, this._mapId);
	SceneManager._scene._spriteset.spawnedEvent(spawnEvent);
	

	
}; 

/**
 * Transform an Event on the GameMap into an Event from a SpawnMap.
 * 
 * @param {number} eventId - EventId from the GameMap to Transform
 * @param {number} mapId - SpawnMap Id
 * @param {number} spawnId - Event Id from the SpawnMap
 */
Game_Map.prototype.transformEvent = function(eventId, mapId, spawnId) {
	var spawnEvent = $gameMap._events[eventId];
	var mapData = $MapData[mapId];
	var event = mapData.events[spawnId];
	var event = JsonEx.makeDeepCopy(event);
	const parentBoundary = !!spawnEvent._parentBoundary ? spawnEvent._parentBoundary : null;
	event.x = spawnEvent.x;
	event.y = spawnEvent.y;
	$dataMap.events[eventId] = event;
	spawnEvent.newData(event);
	if (!this._spawnedList.contains(eventId)) 
		this._spawnedList.push(eventId);
	spawnEvent._spawnMap = mapId;
	spawnEvent._spawnId = spawnId;
	if (this._savedEvents[this._mapId][eventId]) 
		this._savedEvents[this._mapId][eventId] = this.event(eventId);
	if (parentBoundary !== null) spawnEvent._parentBoundary = parentBoundary;
	
	
};

/**
 * Spawn an Event on a Random Tile in a Region.
 * 
 * @param {number} mapId - MapId of a SpawnMap
 * @param {number} eventId - EventId from a SpawnMap
 * @param {array} regions - Array of RegionIds to Spawn On
 * @param {boolean} save - Saved Event Data?
 * @returns 
 */
Game_Map.prototype.spawnEventRegion = function(mapId, eventId, region, save = false) {
	if (!Array.isArray(region)) region = [region];
	coords = Ritter.getRandomTileInRegion(region);
	if (!coords) return;
	Ritter.spawnEvent(mapId, eventId, coords[0], coords[1], save);
};

/**
 * Returns a Random Tile on the GameMap Located Within an Array of RegionIds.
 * 
 * @param {array} regions - RegionsIds to Include when Finding a Tile
 * @returns {array} - [x, y]
 */
Ritter.getRandomTileInRegion = function(regions) {
	mapId = $gameMap._mapId;
	var validTiles = [];
	for (let i = 0; i < Ritter._mapRegionData[mapId].length; i++) {
		if (regions.contains(i)) {
			for (let j = 0; j < Ritter._mapRegionData[mapId][i].length; j++) {
				if (!Ritter.canSpawnOn(Ritter._mapRegionData[mapId][i][j][0],
					Ritter._mapRegionData[mapId][i][j][1])) continue;
				validTiles.push(Ritter._mapRegionData[mapId][i][j]);
			}
		}
	}
	if (validTiles.length === 0) return console.log("Failed to find spawnable tile on regions");
	tile = Math.floor(Math.random() * validTiles.length);
	coords = validTiles[tile];
	return [coords[0], coords[1]];
};

/**
 * Finds The First Available Event For Recycling.
 * 
 * @returns {number} Event Id
 */
Game_Map.prototype.getFirstNullEvent = function() {
	//if ($dataMap.events.length < Ritter.Params.FirstId) $dataMap.events[Ritter.Params.FirstId] = undefined;
	//for (i = Ritter.Params.FirstId; i <= $dataMap.events.length; i++) {
	if ($dataMap.events.length < $gameVariables.value(398)) $dataMap.events[$gameVariables.value(398)] = undefined;
	for (i = $gameVariables.value(398); i <= $dataMap.events.length; i++) {
		if ($dataMap.events[i] === undefined && this._savedEvents[this._mapId][i] === undefined) return i;
	}
};

/**
 * Unspawn an Event.
 * 
 * @param {number} eventId - EventId from the GameMap to Unspawn
 * @param {boolean} removeSave - Remove Saved Event Data?
 */
Game_Map.prototype.unspawnEvent = function(eventId, removeSave) {
	if (!this.event(eventId) || this.event(eventId)._erased) {
		if (Ritter.Params.devMode) {
			let reason = !this.event(eventId) ? "Event Id: " + eventId + " does not exist on the Game Map.\nDid you use the correct Event Id?" : "Event Id: " + eventId + " is already unspawned.\nIt's best not to unspawn events that are already unspawned.\n\nAborting unspawnEvent call.";
			alert(
				"⚡ 𝗥𝗶𝘁𝘁𝗲𝗿_𝗘𝘃𝗲𝗻𝘁𝗦𝗽𝗮𝘄𝗻𝗲𝗿.𝗷𝘀 : 𝗘𝗩𝗘𝗡𝗧 𝗨𝗡𝗦𝗣𝗔𝗪𝗡𝗜𝗡𝗚 : 𝗪𝗔𝗥𝗡𝗜𝗡𝗚! ⚡\n" +
				"🡺 Unable to Unspawn Event. 🡸\n\n" +
				reason
				);
		}
		return;
	}
	mapId = this._mapId;
	index = this._spawnedList.indexOf(eventId);
	this._unspawnList.push(eventId);
	this._unspawnList.sort(function(a, b) {
		return a - b;
	});
	this._spawnedList.splice(index, 1);
	
	if (!this._savedEvents[this._mapId][eventId]) Ritter.resetSelfSwitches(eventId, mapId);
	if (removeSave && this._savedEvents[this._mapId][eventId]) {
		Ritter.resetSelfSwitches(eventId, mapId);
		this.deleteSave(eventId);
	}
	if (this._savedEvents[this._mapId][eventId] === this.event(eventId) && !removeSave) {
		this._savedEvents[this._mapId][eventId] = JsonEx.makeDeepCopy(this.event(eventId));
	}

	this.event(eventId).setPosition(-1, -1);
	this.eraseEvent(eventId);
	if (!!this.event(eventId)._parentBoundary) {
		boundaryName = this.event(eventId)._parentBoundary;
		boundary = Ritter.Boundary.getBoundary(boundaryName);
		idIndex = boundary.boundaryEvents.indexOf(eventId);
		boundary.boundaryEvents.splice(idIndex, 1);
		delete this.event(eventId)._parentBoundary;
	}
	
};

/**
 * Reset an Events Self Switches.
 * 
 * @param {number} eventId - Event Id on the GameMap
 * @param {number} mapId - GameMap MapId
 */
Ritter.resetSelfSwitches = function(eventId, mapId) {
	letters = ['A', 'B', 'C', 'D'];
	letters.forEach((letter) => {
		let key = [mapId, eventId, letter];
		$gameSelfSwitches.setValue(key, false);
	});
};

/**
 * Unspawn All Spawned Events on the Game Map.
 * 
 * @param {boolean} removeSave - Remove All Saved Event Data?
 */
Game_Map.prototype.unspawnAll = function(removeSave = false) {
	//for (let i = Ritter.Params.FirstId; i < $dataMap.events.length; i++) {
	for (let i = $gameVariables.value(398); i < $dataMap.events.length; i++) {
		if (!$gameMap.event(i) || $gameMap.event(i)._erased) continue;
		Ritter.unspawnEvent(i, removeSave);
	}
};

/**
 * Check if Tile is Spawnable.
 * 
 * @param {number} x - X Location
 * @param {number} y - Y Location
 * @param {array} regions - (Optional) Valid RegionIds
 * @returns true if valid spawn tile
 */
Ritter.canSpawnOn = function(x, y, regions) {
	var region = $gameMap.regionId(x,y);
	if ($gameMap.eventsXy(x, y).length > 0) return false;
	if ($gamePlayer.x == x && $gamePlayer.y == y) return false;
	if (regions && !regions.contains(region)) return false;
	//if (!$gameMap.isPassable(x,y)) return false;
	return true;
};

/**
 * --------------------------------------------------------------------------------
 *   Saved Event Code
 * --------------------------------------------------------------------------------
 */

/**
 * Update/Create Saved Event Data.
 * 
 * @param {number} eventId - Event Id to save
 * @param {number} mapId - Map Id tied to Event
 */
Game_Map.prototype.saveEvent = function(eventId, mapId) {
	if (!this._savedEvents) this._savedEvents = [];
	if (!this._savedEvents[this._mapId]) this._savedEvents[this._mapId] = [];
	this._savedEvents[this._mapId][eventId] = this.event(eventId);
};

/**
 * Delete Saved Event Data.
 * 
 * @param {number} eventId - Event Id to Delete Save Data From
 */
Game_Map.prototype.deleteSave = function(eventId) {
	if (!this._savedEvents[this._mapId][eventId]) { 
		Alert("⚡ 𝗥𝗶𝘁𝘁𝗲𝗿_𝗘𝘃𝗲𝗻𝘁𝗦𝗽𝗮𝘄𝗻𝗲𝗿.𝗷𝘀 : 𝗘𝘃𝗲𝗻𝘁 𝗦𝗮𝘃𝗲 𝗗𝗮𝘁𝗮 𝗗𝗲𝗹𝗲𝘁𝗶𝗼𝗻 : 𝗪𝗔𝗥𝗡𝗜𝗡𝗚! ⚡\n" +
		"🡺 Event Id Not Found! 🡸\n\n" +
		"There is no Saved Event Data for:\n" +
		"EventId: " + eventId + " on Map: " + this._mapId + "\n\n" +
		"Did you use the Correct Event Id?\n\n" +
		"Disable Unspawn Warning 3 In Plugin Parameters To Never See This Warning Again."
		);
		return; 
	};
	delete this._savedEvents[this._mapId][eventId];
};

Game_Map.prototype.restoreUnspawnedEvent = function(eventId) {
	let event = this._savedEvents[this._mapId][eventId];
	this._events[eventId] = this._savedEvents[this._mapId][eventId];
	this.event(eventId).setPosition(event.x, event.y);
	SceneManager._scene._spriteset.spawnedEvent(this.event(eventId));
	this._spawnedList.push(eventId);
};

/**
 * --------------------------------------------------------------------------------
 *   Event Setup Code
 * --------------------------------------------------------------------------------
 */

/**
 * Handles Restoring Saved Events.
 */
Game_Map.prototype.restoreSavedEvents = function() {
    //for (var i = Ritter.Params.FirstId; i < this._savedEvents[this._mapId].length; i++) {
	for (var i = $gameVariables.value(398); i < this._savedEvents[this._mapId].length; i++) {
        if (!this._savedEvents[this._mapId][i]) continue;
        let event = this._savedEvents[this._mapId][i];
        if (!(event instanceof Game_Event)) { 
            continue;
        }
        
		// This never had problems before but for some reason in this one circumstance at least
		// there is a problem. We'll just use the eventData that is stored on each event instead.
		// it's the same data anyway.
		// Going to leave it here in case it's useful later.
		//$dataMap.events[i] = JsonEx.makeDeepCopy($MapData[event._spawnMap].events[event._spawnId]);
		
		$dataMap.events[i] = JsonEx.makeDeepCopy(event._eventData);
        
		$dataMap.events[i].x = event._x;
        $dataMap.events[i].y = event._y;
        $dataMap.events[i].id = event._eventId;
    }
};

Ritter_Game_Map_setupEvents = Game_Map.prototype.setupEvents;
Game_Map.prototype.setupEvents = function() {
	if (!this._savedEvents) this._savedEvents = [];
	if (!this._savedEvents[this._mapId]) 
		this._savedEvents[this._mapId] = [];
	if (this._savedEvents[this._mapId].length > 0) 
		this.restoreSavedEvents();
	
	if (Imported.Ritter_BoundarySystem) {
		Ritter.Boundary.clearAllBoundaryEvents();
	}
	
	this._spawnedList = [];
	this._unspawnList = [];
	if (Imported.Ritter_BoundarySystem) {
		Ritter.Boundary.initBoundaries();
	}
	$MapData[this._mapId].sprites = [];
	Ritter_Game_Map_setupEvents.call(this);
	if (this._savedEvents[this._mapId].length > 0) this.postSetup();
};

/**
 * Initializes a Boundarys Saved Events.
 * Respawns saved events found within a boundary when called.
 * 
 * @param {string} boundaryName - The "Name" of a Boundary.
 */
Ritter.Boundary.initBoundaryEvents = function(boundaryName) {
	if (!this.activeBoundaries.contains(boundaryName)) return;
	const boundary = this.getBoundary(boundaryName);
	const tiles = this.getInBoundaryTiles(boundary);
	
	for (let i = 0; i < tiles.length; i++) {
		let tile = tiles[i];
		if (tile[0] == $gamePlayer.x && tile[1] == $gamePlayer.y) continue;
		if (tile[0] < 0 || tile[0] > $dataMap.width - 1) continue;
		if (tile[1] < 0 || tile[1] > $dataMap.height - 1) continue;
		if ($gameMap.eventIdXy(tile[0], tile[1]) > 0) continue;
		if (boundary.boundaryEvents.length >= boundary.maxEvents) continue;
		if (boundary.savedEvents[$gameMap._mapId][tile[0]][tile[1]] === null) continue;
		
		$gameMap.restoreUnspawnedSavedBoundaryEvent(boundary, tile[0], tile[1]);
	}
};

/**
 * Restores saved events after Game_Map.prototype.setupEvents runs.
 */
Game_Map.prototype.postSetup = function() {
	//for (var i = Ritter.Params.FirstId; i < this._savedEvents[this._mapId].length; i++) {
	for (var i = $gameVariables.value(398); i < this._savedEvents[this._mapId].length; i++) {
		if (this._savedEvents[this._mapId][i] === undefined) continue;
		event = this._savedEvents[this._mapId][i];
		this._events[i] = event;
		this._spawnedList.push(i);
	}
};

/**
 * Adding calls to update saved Events before map change.
 */
Ritter_Game_Player_reserveTransfer = Game_Player.prototype.reserveTransfer;
Game_Player.prototype.reserveTransfer = function(mapId, x, y, d, fadeType) {
	if (!$gameMap._savedEvents) $gameMap._savedEvents = {};
	if (!$gameMap._savedEvents[$gameMap._mapId]) $gameMap._savedEvents[$gameMap._mapId] = [];
	if ($gameMap._savedEvents[$gameMap._mapId].length > 0) $gameMap.updateSavedEvents();
	if (Imported.Ritter_BoundarySystem) $gameMap.updateBoundarySavedEvents();

	Ritter_Game_Player_reserveTransfer.call(this,mapId, x, y, d, fadeType);
};

/**
 * Updates all saved event data.
 */
Game_Map.prototype.updateSavedEvents = function() {
	map = $gameMap;
	//for (var i = Ritter.Params.FirstId; i < map._savedEvents[map._mapId].length; i++) {
	for (var i = $gameVariables.value(398); i < map._savedEvents[map._mapId].length; i++) {
		if (map._savedEvents[map._mapId][i] === undefined) continue;
		if (!map.event(i)) continue;
		if (map.event(i).x === -1) continue;
		this.saveEvent(map.event(i));
	}
};

/**
 * Create Spawn Event Character Sprite and Remove Old Sprite.
 * 
 * @param {object} event - $gameMap Event
 */
Spriteset_Map.prototype.spawnedEvent = function(event) {
	if (!this._characterSprites) return;
	if (!$MapData[event._mapId].sprites) $MapData[event._mapId].sprites = [];
	if (!!$MapData[event._mapId].sprites[event._eventId]) {
		const sprite = $MapData[event._mapId].sprites[event._eventId];
		const index = this._characterSprites.indexOf(sprite);
		this._tilemap.removeChild(sprite);
		this._characterSprites.splice(index, 1);
	}
	const eventSprite = new Sprite_Character(event);
	this._characterSprites.push(eventSprite);
	this._tilemap.addChild(eventSprite);
	
	$MapData[event._mapId].sprites[event._eventId] = eventSprite;
};

/**
 * Inserting Ritter.recreateDatamapEvents() into Spriteset_Map.createCharacters
 */
Ritter_Spriteset_Map_createCharacters = Spriteset_Map.prototype.createCharacters;
Spriteset_Map.prototype.createCharacters = function() {	
	if ($gameMap._spawnedList.length > 0) {
		//if ($gameMap.event(Ritter.Params.FirstId) !== undefined && $dataMap.events[Ritter.Params.FirstId] == undefined)  {
		if ($gameMap.event($gameVariables.value(398)) !== undefined && $dataMap.events[$gameVariables.value(398)] == undefined)  {
			Ritter.recreateDatamapEvents();
		}
	}
	Ritter_Spriteset_Map_createCharacters.call(this);
};

/**
 * Recreate the $dataMap.events data for spawned events on scene change.
 */
Ritter.recreateDatamapEvents = function() {
	//const end = $gameMap._spawnedList.length + $gameMap._unspawnList.length + Ritter.Params.FirstId;
	//for (let i = Ritter.Params.FirstId; i < end; i++) {
	const end = $gameMap._spawnedList.length + $gameMap._unspawnList.length + $gameVariables.value(398);
	for (let i = $gameVariables.value(398); i < end; i++) {
		if (!$gameMap._events[i]) continue;
		$dataMap.events[i] = $gameMap._events[i]._eventData;
		$dataMap.events[i].x = $gameMap._events[i].x;
		$dataMap.events[i].y = $gameMap._events[i].y;
	}
};

/**
	 * Extract required data from every map on game boot.
	 * Build $MapData Database.
	 */
 Ritter._initMapData = function() {
	for (let i = 1; i < $dataMapInfos.length; i++) {
		if (!$dataMapInfos[i]) continue;
		const map = new XMLHttpRequest();
		const url = "data/Map%1.json".format(i.padZero(3));
		map.open("GET", url);
		map.overrideMimeType("application/json");
		map.onload = function() {
			const mapData = JSON.parse(map.responseText);
			$MapData[i] = {};
			$MapData[i].mapId = i;
			$MapData[i].width = mapData.width;
			$MapData[i].height = mapData.height;
			$MapData[i].sprites = [];
			$MapData[i].events = [];
		}
		map.onerror = function() { throw new Error('Failed to load: ' + url); };
		map.send();
	}
};

/**
 * Initialize Boundary Database after Game Databases load on boot.
 */
const Ritter_Scene_Boot_onDatabaseLoaded = Scene_Boot.prototype.onDatabaseLoaded;
Scene_Boot.prototype.onDatabaseLoaded = function() {
	Ritter_Scene_Boot_onDatabaseLoaded.call(this);
	Ritter._initMapData();
	DataManager.preloadMaps();
};

/** ------------------------------------------------------------------------------------
 *   Boundary System Compatibility
 *   Requires a Separate Plugin to Utilize These Features.
 * 
 *   Allows for the Spawning and Unspawning of recycled events on a boundary around
 *   the player, an event, or an x,y location.
 * 
 *   Updated to provide Automatic Boundaries.
 * 
 *   View store page for Ritter Ultimate Event Spawner for Instructions.
 * 
 *   Extension plugin available on link below.
 * 
 *   Plugin Store Page: https://notritter.itch.io/ritter-boundary-system-rpg-maker-mz
 *  ------------------------------------------------------------------------------------
 */

// Get to fixing tabbing here without messing up the code next time.
//if (Imported.Ritter_BoundarySystem) {
	
	Ritter.Boundary._boundaryType = [];
	Ritter.Boundary.activeBoundaries = [];

	/**
	 * Inserting Handler for Automatic Boundaries into Game_Map update.
	 */
	const Ritter_Game_Map_update = Game_Map.prototype.update;
	Game_Map.prototype.update = function(sceneActive) {
    	Ritter_Game_Map_update.call(this, sceneActive);
		if (Ritter.Boundary.activeBoundaries.length > 0)  { 
			Ritter.Boundary.autoBoundary();
		}
	};

	/**
	 * Handles All Active Boundaries Once Every Update.
	 */
	Ritter.Boundary.autoBoundary = function() {
		for (let i = 0; i < this.activeBoundaries.length; i++) {
			// Get boundary object using activeBoundaries[i]
			const boundary = this.getBoundary(this.activeBoundaries[i]);
			// checks if boundary is a spawn boundary and checks for any saved event data on/in the boundary.
			if (boundary.type.toLowerCase() === "spawnon" || boundary.type.toLowerCase() === "spawnin" || boundary.type.toLowerCase() === "fillon" || boundary.type.toLowerCase() === "fillin") {
				this.checkBoundaryTileSaveData(boundary);
			}
			// adjust wait time tick
			boundary.tick += 1;
			
			// if tick < wait time then do nothing.
			if (boundary.tick < boundary.waittime) continue;
			
			// tick is >= wait setting value to 0 and spawning an event.
			boundary.tick = 0;
			if (boundary.boundaryEvents.length >= boundary.maxEvents && !(boundary.type.toLowerCase().contains("unspawn"))) continue;
			
			if (!!boundary.maps[$gameMap._mapId]) {
				if (boundary.type.toLowerCase().contains("unspawn")) {
					switch (boundary.type.toLowerCase()) {
						case "unspawnon":
							Ritter.unspawnEventOnBoundary(boundary.name);
							break;
						case "unspawnin":
							Ritter.unspawnEventInBoundary(boundary.name);
							break;
						default:
							break;
					}
					continue; // Exits 'Unspawn' Boundaries here.
				}

				var regions = [];
				const end = boundary.maps[$gameMap._mapId].regions.length;
				for (let regionId = 0; regionId < end; regionId++) {
					if (!!boundary.maps[$gameMap._mapId].regions[regionId]) {
						if (!regions.contains(regionId)) {
							regions.push(regionId);
						}
					}
				}
				var coords;
				switch (boundary.type.toLowerCase()) {
					case "spawnon":
						coords = this.randomRegionOnBoundary(regions, boundary);
						break;
					case "spawnin":
						coords = this.randomRegionInBoundary(regions, boundary);
						break;
					case "fillon":
						Ritter.Boundary.autoFillBoundary(boundary);
						break;
					case "fillin":
						Ritter.Boundary.autoFillBoundary(boundary);
						break;
					default:
						break;
				}
				
				if (!coords) continue; // Exits 'FillIn' + 'FillOn' boundaries here.
				const region = $gameMap.regionId(coords[0], coords[1]);
				let eventArray = boundary.maps[$gameMap._mapId].regions[region];
				let events = this.findSpawnableEvents(eventArray, boundary);
				if (!events) continue;
				const eventId = events[Math.floor(Math.random()*events.length)];
				const save = "auto";

				if (!boundary.savedEvents[$gameMap._mapId][coords[0]])
					continue;
				if (boundary.savedEvents[$gameMap._mapId][coords[0]][coords[1]] != null) return;
				$gameMap.spawnBoundaryEvent(boundary.spawnMap, eventId, coords[0], coords[1], boundary.name, save);
			}
		}
	};

	/**
	 * Fills every valid tile 'On' or 'In' a Boundary with events.
	 * 
	 * @param {object} boundary - Boundary Data Object
	 */
	Ritter.Boundary.autoFillBoundary = function(boundary) {
		const tiles = boundary.type.toLowerCase() === "fillon" ? this.getOnBoundaryTiles(boundary) : this.getInBoundaryTiles(boundary);
		for (let i = 0; i < tiles.length; i++) {
			let tile = tiles[i];
			// No spawning on player
			if (tile[0] == $gamePlayer.x && tile[1] == $gamePlayer.y) continue;
			// No spawning outside the map
			if (tile[0] < 0 || tile[0] > $dataMap.width - 1) continue;
			if (tile[1] < 0 || tile[1] > $dataMap.height - 1) continue;
			// No spawning on other events
			if ($gameMap.eventIdXy(tile[0], tile[1]) > 0) continue;
			// No spawning if the boundary is at max events
			if (boundary.boundaryEvents.length >= boundary.maxEvents) continue;
			
			let mapId = $gameMap._mapId;
			let regionId = $gameMap.regionId(tile[0], tile[1]);
			let eventArray = boundary.maps[mapId].regions[regionId];
			let events = this.findSpawnableEvents(eventArray, boundary);
			if (!events) continue;
			let eventId = events[Math.floor(Math.random()*events.length)];
			let mapData = $MapData[boundary.spawnMap];
			let save = mapData.events[eventId].BoundaryData.SavedEvent;

			$gameMap.spawnBoundaryEvent(boundary.spawnMap, eventId, tile[0], tile[1], boundary.name, save);
		}
	}

	/**
	 * Checks events array for Enabled Events and returns an Array 
	 * of Spawnable Events.
	 * 
	 * @param {array} events - An array of Event Ids.
	 * @param {object} boundary - A Boundary Object.
	 * @returns - An array of Enabled Events.
	 */
	Ritter.Boundary.findSpawnableEvents = function(events, boundary) {
		if (!events) return;
		let eventIds = [];
		let spawnMap = boundary.spawnMap;
		let mapData = $MapData[spawnMap];
		events.forEach((eventId) => {
			let event = mapData.events[eventId];
			if (event.BoundaryData.Enabled) eventIds.push(eventId);
		});
		return eventIds.length > 0 ? eventIds : false;
	}

	/**
	 * Enables an Event so that it may spawn on Auto Boundaries.
	 * 
	 * @param {number} spawnMap - Spawn Map Id #
	 * @param {number} eventId - Spawn Map Event Id #
	 */
	Ritter.Boundary.enableEvent = function(spawnMap, eventId) {
		const mapData = $MapData[spawnMap];
		if (!mapData) return;
		const boundaryData = mapData.events[eventId].BoundaryData;
		if (!boundaryData.Enabled) {
			boundaryData.Enabled = true;
		}
	}

	/**
	 * Disables an Event so that it will no longer spawn on Auto Boundaries.
	 * 
	 * @param {number} spawnMap - Spawn Map Id #
	 * @param {number} eventId - Spawn Map Event Id #
	 */
	Ritter.Boundary.disableEvent = function(spawnMap, eventId) {
		const mapData = $MapData[spawnMap];
		if (!mapData) return;
		const boundaryData = mapData.events[eventId].BoundaryData;
		if (boundaryData.Enabled) {
			boundaryData.Enabled = false;
		}
	}

	/**
	 * Check every tile On/In a Spawn Boundary for saved event data
	 * and spawn an event on a tile if save data is found.
	 * 
	 * @param {object} boundary - Boundary Object.
	 */
	Ritter.Boundary.checkBoundaryTileSaveData = function(boundary) {
		const tiles = boundary.type.toLowerCase() === "spawnon" || boundary.type.toLowerCase() === "fillon" ? this.getOnBoundaryTiles(boundary) : this.getInBoundaryTiles(boundary);
		for (let i = 0; i < tiles.length; i++) {
			let tile = tiles[i];
			if (tile[0] == $gamePlayer.x && tile[1] == $gamePlayer.y) continue;
			if (tile[0] < 0 || tile[0] > $dataMap.width - 1) continue;
			if (tile[1] < 0 || tile[1] > $dataMap.height - 1) continue;
			
			// This line here causes a ton of delay.
			// drops framerate when called a lot.
			// need another solution for $gameMap.eventIdXy() that isn't so slow.
			//if ($gameMap.eventIdXy(tile[0], tile[1]) > 0) continue;
			
			// This will look at boundary max events number and compare it to
			// boundary events which are active (spawned) and if the boundary
			// has spawned the maximum number of events it will 
			// clear the saved event data for that tile and continue
			// continue will not respawn the saved event.
			// if you want different functionality other than that then 
			// this is where to look.
			if (boundary.boundaryEvents.length >= boundary.maxEvents) {
				if (boundary.savedEvents[$gameMap._mapId][tile[0]][tile[1]] != null) {
				boundary.savedEvents[$gameMap._mapId][tile[0]][tile[1]] = null;
				continue;
				}
			}
			
			// this will continue if there is no saved event to restore.
			if (boundary.savedEvents[$gameMap._mapId][tile[0]][tile[1]] === null) 
				continue;
			
			// Maybe there will be less lag putting it down here?
			// This way it will only check if there's an event on the tile
			// if there is actually saved event data for that tile first?
			// Still though people might want to spawn saved events over other events.
			if ($gameMap.eventIdXy(tile[0], tile[1]) > 0) 
				continue;
			
			$gameMap.restoreUnspawnedSavedBoundaryEvent(boundary, tile[0], tile[1]);
		}
	};
	
	/**
	 * Spawn an Event On the Edge of a Boundary.
	 * 
	 * @param {number} mapId - Spawn Map Id.
	 * @param {number} eventId - Event Id from Spawn Map.
	 * @param {array} regions - Array of Valid RegionIds to Spawn On.
	 * @param {string} boundaryName - Name of a Boundary to Spawn On.
	 * @returns 
	 */
	Ritter.spawnEventOnBoundary = function(mapId, eventId, regions, boundaryName, save) {
		const boundary = Ritter.Boundary.getBoundary(boundaryName);
		if (!(boundary.boundaryEvents.length < boundary.maxEvents)) return;
		const coords = Ritter.Boundary.randomRegionOnBoundary(regions, boundary);
		if (!coords) return;
		$gameMap.spawnBoundaryEvent(mapId, eventId, coords[0], coords[1], boundaryName, save);
	};

	/**
	 * Handles Spawning a Boundary Event on an x, y Location.
	 * 
	 * @param {number} mapId - Spawn Map Id.
	 * @param {number} eventId - Event Id from Spawn Map
	 * @param {number} x - X Coordinate to Spawn On.
	 * @param {number} y - Y Coordinate to Spawn On.
	 * @param {string} boundaryName - Name of a Boundary to Spawn on.
	 */
	Game_Map.prototype.spawnBoundaryEvent = function(mapId, eventId, x, y, boundaryName, save) {
		if (!(SceneManager._scene instanceof Scene_Map)) return;
		if (!Ritter.canSpawnOn(x, y)) return;
		if (!this._unspawnList) this._unspawnList = [];
		if (!this._spawnedList) this._spawnedList = [];
		const boundary = Ritter.Boundary.getBoundary(boundaryName);
		const nextId = this.getFirstNullEvent();
		const mapData = $MapData[mapId];
		const event = JsonEx.makeDeepCopy(mapData.events[eventId]);
		event.x = x;
		event.y = y;
		if (save === "auto") save = event.BoundaryData.SavedEvent;
		
		if (this._unspawnList.length > 0) {
			var spawnEvent = $gameMap._events[this._unspawnList[0]];
			event.id = this._unspawnList[0];
			$dataMap.events[this._unspawnList[0]] = event;
			spawnEvent.newData(event);
			this._spawnedList.push(this._unspawnList[0]);
			this._unspawnList.splice(0, 1);
		} else {
			event.id = nextId;
			$dataMap.events[nextId] = event;
			var spawnEvent = new Game_Event(this._mapId, nextId);
			spawnEvent.newData(event);
			this._events[nextId] = spawnEvent;
			this._spawnedList.push(nextId);
		}
		this._lastSpawnEventId = spawnEvent._eventId;
		spawnEvent._spawnMap = mapId;
		spawnEvent._spawnId = eventId;
		spawnEvent._parentBoundary = boundaryName;
		spawnEvent._savedEvent = save;
		boundary.boundaryEvents.push(event.id);
		SceneManager._scene._spriteset.spawnedEvent(spawnEvent);
	}; 
	
	/**
	 * Spawn an Event On a Random Region Tile On the Edge of a Boundary
	 * 
	 * @param {number} mapId - Spawn Map Id.
	 * @param {number} eventId - Event Id from Spawn Map
	 * @param {array} regions - Array of Valid RegionIds to Spawn on.
	 * @param {string} boundaryName - Name of a Boundary to Spawn on.
	 */
	Ritter.spawnEventOnBoundary = function(mapId, eventId, regions, boundaryName) {
		const boundary = Ritter.Boundary.getBoundary(boundaryName);
		if (!(boundary.boundaryEvents.length < boundary.maxEvents)) return;
		const coords = Ritter.Boundary.randomRegionOnBoundary(regions, boundary);
		if (!coords) return;
		$gameMap.spawnBoundaryEvent(mapId, eventId, coords[0], coords[1], boundaryName);
	};
	
	/**
	 * Spawn an Event On a Random Region Tile Within a Boundary.
	 * 
	 * @param {number} mapId - Spawn Map Id.
	 * @param {number} eventId - Event Id from Spawn Map
	 * @param {array} regions - Array of Valid RegionIds to Spawn on.
	 * @param {string} boundaryName - Name of a Boundary to Spawn on.
	 */
	Ritter.spawnEventInBoundary = function(mapId, eventId, regions, boundaryName) {
		const boundary = Ritter.Boundary.getBoundary(boundaryName);
		if (!(boundary.boundaryEvents.length < boundary.maxEvents)) return;
		const coords = Ritter.Boundary.randomRegionInBoundary(regions, boundary);
		if (!coords) return;
		$gameMap.spawnBoundaryEvent(mapId, eventId, coords[0], coords[1], boundaryName);
	};
	
	/**
	 * Fill every tile within a Boundary with an Event.
	 * 
	 * @param {number} mapId - Spawn Map Id.
	 * @param {number} eventId - Event Id from Spawn Map
	 * @param {array} regions - Array of Valid RegionIds to Spawn on.
	 * @param {string} boundaryName - Name of a Boundary to Spawn on.
	 */
	Ritter.spawnFillInBoundary = function(mapId, eventId, regions, boundaryName) {
		if (!Array.isArray(regions)) regions = [regions];
		const boundary = Ritter.Boundary.getBoundary(boundaryName);
		const tiles = Ritter.Boundary.getInBoundaryTiles(boundary);
	
		tiles.forEach((tile) => {
			if (!(tile[0] == $gamePlayer.x && tile[1] == $gamePlayer.y)
				&& (tile[0] >= 0 && tile[0] <= $dataMap.width - 1)
				&& (tile[1] >= 0 && tile[1] <= $dataMap.height - 1)
				&& !$gameMap.eventIdXy(tile[0], tile[1])
				&& boundary.boundaryEvents.length < boundary.maxEvents
				&& regions.contains($gameMap.regionId(tile[0], tile[1]))) {
				$gameMap.spawnBoundaryEvent(mapId, eventId, tile[0], tile[1], boundaryName);
			}
		});
	};

	/**
	 * Fill every tile on the Edge of a Boundary with an Event.
	 * 
	 * @param {number} mapId - Spawn Map Id.
	 * @param {number} eventId - Event Id from Spawn Map
	 * @param {array} regions - Array of Valid RegionIds to Spawn on.
	 * @param {string} boundaryName - Name of a Boundary to Spawn on.
	 */
	Ritter.spawnFillOnBoundary = function(mapId, eventId, regions, boundaryName) {
		if (!Array.isArray(regions)) regions = [regions];
		const boundary = Ritter.Boundary.getBoundary(boundaryName);
		const tiles = Ritter.Boundary.getOnBoundaryTiles(boundary);
	
		tiles.forEach((tile) => {
			if (!(tile[0] == $gamePlayer.x && tile[1] == $gamePlayer.y)
				&& (tile[0] >= 0 && tile[0] <= $dataMap.width - 1)
				&& (tile[1] >= 0 && tile[1] <= $dataMap.height - 1)
				&& !$gameMap.eventIdXy(tile[0], tile[1])
				&& boundary.boundaryEvents.length < boundary.maxEvents
				&& regions.contains($gameMap.regionId(tile[0], tile[1]))) {
				$gameMap.spawnBoundaryEvent(mapId, eventId, tile[0], tile[1], boundaryName);
			}
		});
	};

	/**
	 * Unspawn an Event On the Edge of a Boundary. ◻
	 * 
	 * @param {string} boundaryName - Name of an Unspawn Boundary.
	 */
	Ritter.unspawnEventOnBoundary = function(boundaryName) {
		const boundary = Ritter.Boundary.getBoundary(boundaryName);
		const tiles = Ritter.Boundary.getOnBoundaryTiles(boundary);
		for (let i = 0; i < tiles.length; i++) {
			let eventId = $gameMap.eventIdXy(tiles[i][0], tiles[i][1]);
			if (!$gameMap.event(eventId)) continue;
			if ($gameMap.event(eventId)._parentBoundary == "") continue;
			if (boundary.BoundaryList == null) boundary.BoundaryList = [];
			if (!boundary.BoundaryList.contains($gameMap.event(eventId)._parentBoundary)) continue;
			let event = $gameMap.event(eventId);
			let eventBoundary = Ritter.Boundary.getBoundary(event._parentBoundary);
			
			if (eventBoundary.boundaryEvents.contains(eventId)) {
				if (event._eventData.BoundaryData.SavedEvent) {
					const x = tiles[i][0];
					const y = tiles[i][1];
					const mId = $gameMap._mapId;
					event._eventData.x = tiles[i][0];
					event._eventData.y = tiles[i][1];
					eventBoundary.savedEvents[mId][x][y] = JsonEx.makeDeepCopy(event);
					Ritter.Boundary.storeSelfSwitches(eventBoundary, $gameMap.event(eventId));
				}
				Ritter.unspawnEvent(eventId);
			}
		}
	};

	Ritter.Boundary.unspawnBoundaryEvent = function(eventId, removeSave = false) {
		const event = $gameMap.event(eventId);
		const boundaryName = event._parentBoundary;
		if (boundaryName === "") return;
		const boundary = Ritter.Boundary.getBoundary(boundaryName);
		if (!boundary.boundaryEvents.contains(eventId)) return;
		if (removeSave && event._eventData.BoundaryData.SavedEvent) {
			event._eventData.BoundaryData.SavedEvent = false;
			event._savedEvent = false;
		}
		Ritter.unspawnEvent(eventId);
	};

	
	/**
	 * Removes All Saved Data For This Spawn Map Event From Boundary Saved Data
	 * On A Map.
	 * 
	* @param {string} boundaryName - The Boundary To Remove Saved Data From.
	* @param {number} mapId - The GameMap To Remove Saved Data From.
	* @param {number} eventId - Event Id from Spawn Map tied to Boundary.
	*/
	Ritter.Boundary.removeSaves = function(boundaryName, mapId, eventId) {
		const boundary = this.getBoundary(boundaryName);
		const saveData = boundary.savedEvents[mapId];
		for (let x = 0; x < saveData.length; x++) {
			for (let y = 0; y < saveData[x].length; y++) {
				if (saveData[x][y] == null) continue;
				if (saveData[x][y]._spawnId != eventId) continue;
				saveData[x][y] = null;
			}
		}
	}

	/**
	 * Deletes Saved Event Data Using x,y Coordinates.
	 * 
	 * @param {number} x - The X Location of a Saved Event.
	 * @param {number} y - The Y Location of a Saved Event.
	 * @param {string} boundaryName - The Name of a Boundary.
	 * @param {number} mapId - Optional - Map Id #. Defaults to $gameMap._mapId
	 */
	Ritter.Boundary.removeSaveXy = function(x, y, boundaryName, mapId = $gameMap._mapId) {
		const boundary = this.getBoundary(boundaryName);
		if (boundary.savedEvents[mapId][x][y] != null) 
			boundary.savedEvents[mapId][x][y] = null;
	}
	
	/** 
	 * Unspawn an Event Within a Boundary. ◼
	 * 
	 * @param {string} boundaryName - Name of an Unspawn Boundary.
	 */
	Ritter.unspawnEventInBoundary = function(boundaryName) {
		const boundary = Ritter.Boundary.getBoundary(boundaryName);
		const tiles = Ritter.Boundary.getInBoundaryTiles(boundary);
		for (let i = 0; i < tiles.length; i++) {
			let eventId = $gameMap.eventIdXy(tiles[i][0], tiles[i][1]);
			if (!$gameMap.event(eventId)) continue;
			if ($gameMap.event(eventId)._parentBoundary == "") continue;
			if (!boundary.BoundaryList.contains($gameMap.event(eventId)._parentBoundary)) continue;
			let event = $gameMap.event(eventId);
			let eventBoundary = Ritter.Boundary.getBoundary(event._parentBoundary);
			
			if (eventBoundary.boundaryEvents.contains(eventId)) {
				if (event._eventData.BoundaryData.SavedEvent) {
					const x = event.x;
					const y = event.y;
					const mId = $gameMap._mapId;
					event._eventData.x = tiles[i][0];
					event._eventData.y = tiles[i][1];
					eventBoundary.savedEvents[mId][x][y] = JsonEx.makeDeepCopy(event);
					Ritter.Boundary.storeSelfSwitches(eventBoundary, $gameMap.event(eventId));
				}
				Ritter.unspawnEvent(eventId);
			}
		}
	};

	/**
	 * Unspawn a Boundary Event (and remove save data);
	 * 
	 * @param {number} eventId - Id of Event to Unspawn
	 * @param {boolean} removeSave - Delete Save Data? true/false
	 * @returns 
	 */
	Ritter.unspawnBoundaryEvent = function(eventId, removeSave = false) {
		const event = $gameMap.event(eventId);
		const boundary = Ritter.Boundary.getBoundary(event._parentBoundary);

		if (!event || !boundary) return;
		if (boundary.boundaryEvents.contains(eventId)) {
			if (event._eventData.BoundaryData.SavedEvent && removeSave) {
				const x = event.x;
				const y = event.y;
				const mId = $gameMap._mapId;
				boundary.savedEvents[mId][x][y] = null;
			}
			Ritter.unspawnEvent(eventId);
		}
	}
	
	const Ritter_Boundary_createBoundary = Ritter.Boundary.createBoundary;
	/**
     * Extends Boundary System createBoundary function to work with Spawner.
     * 
     * @param {number} width - Number of Tiles for Boundary Width.
     * @param {number} height - Number of Tiles for Boundary Height.
	 * @param {number} thickness - Number of Tiles for Boundary Thickness.
	 * @param {string} name - Name of Boundary.
	 * @param {number} eventId - Target for Boundary. Use: -1 for x,y location OR 0 for Player OR EventId for Events.
	 * @param {number} expandBy - Number of Tiles the Boundary Expands in the Direction the Player is moving.
	 * @param {number} maxEvents - Maximum Number of Events a Boundary is Allowed to Spawn at once.
	 * @param {number} centerx - X location for Boundary Center. Leave undefined if not spawning on coordinates.
	 * @param {number} centery - Y location for Boundary Center. Leave undefined if not spawning on coordinates.
     */
	Ritter.Boundary.createSpawnerBoundary = function(width, height, thickness, name = '', eventId = -1, expandBy = 4, maxEvents, centerx = undefined, centery = undefined) {
		let abort = false;
		let dev = Ritter.Params.devMode;
		if (!!Ritter.Boundary._boundaryType) {
			for (let i = 0; i < Ritter.Boundary._boundaryType.length; i++) {
				if (Ritter.Boundary._boundaryType[i].name === name) {
					if (dev) alert(
						"⚡ 𝗥𝗶𝘁𝘁𝗲𝗿_𝗘𝘃𝗲𝗻𝘁𝗦𝗽𝗮𝘄𝗻𝗲𝗿.𝗷𝘀 : 𝗕𝗼𝘂𝗻𝗱𝗮𝗿𝘆 𝗖𝗿𝗲𝗮𝘁𝗶𝗼𝗻 : 𝗨𝗦𝗘𝗥 𝗘𝗥𝗥𝗢𝗥! ⚡\n" +
						"🡺 Duplicate Boundary Name Detected! 🡸\n\n" +
						"Boundary Name: '" + name + "' used more than once.\n" +
						"Using duplicate names will cause problems.\n\n" +
						"Aborting Boundary Creation!"
						);
					abort = true;
				}	
			}
		}
		if (width % 2 == 0) {
			if (dev) alert(
			"⚡ 𝗥𝗶𝘁𝘁𝗲𝗿_𝗘𝘃𝗲𝗻𝘁𝗦𝗽𝗮𝘄𝗻𝗲𝗿.𝗷𝘀 : 𝗕𝗼𝘂𝗻𝗱𝗮𝗿𝘆 𝗖𝗿𝗲𝗮𝘁𝗶𝗼𝗻 : 𝗪𝗔𝗥𝗡𝗜𝗡𝗚! ⚡\n" +
			"🡺 An EVEN Number was Used For Boundary Width! 🡸\n\n" +
			"Boundary Named: '" + name + "' has a Width of " + width + "\n" +
			"It is best to use an ODD number for Boundary Width & Height.\n" +
			"This allows for the Center of the boundary to be located on a tile.\n\n" +
			"Try using " + (width - 1) + " or " + (width + 1) + " for better results."
			);
		}
		if (height % 2 == 0) {
			if (dev) alert(
			"⚡ 𝗥𝗶𝘁𝘁𝗲𝗿_𝗘𝘃𝗲𝗻𝘁𝗦𝗽𝗮𝘄𝗻𝗲𝗿.𝗷𝘀 : 𝗕𝗼𝘂𝗻𝗱𝗮𝗿𝘆 𝗖𝗿𝗲𝗮𝘁𝗶𝗼𝗻 : 𝗪𝗔𝗥𝗡𝗜𝗡𝗚! ⚡\n" +
			"🡺 An EVEN Number was Used For Boundary Height! 🡸\n\n" +
			"Boundary Named: '" + name + "' has a Height of " + height + "\n" +
			"It is best to use an ODD number for Boundary Width & Height.\n" +
			"This allows for the Center of the boundary to be located on a tile.\n\n" +
			"Try using " + (height - 1) + " or " + (height + 1) + " for better results."
			);
		}
		if (thickness < 1) {
			if (dev) alert(
				"⚡ 𝗥𝗶𝘁𝘁𝗲𝗿_𝗘𝘃𝗲𝗻𝘁𝗦𝗽𝗮𝘄𝗻𝗲𝗿.𝗷𝘀 : 𝗕𝗼𝘂𝗻𝗱𝗮𝗿𝘆 𝗖𝗿𝗲𝗮𝘁𝗶𝗼𝗻 : 𝗪𝗔𝗥𝗡𝗜𝗡𝗚! ⚡\n" +
				"🡺 Invalid Boundary Thickness! 🡸\n\n" +
				"Boundary Named: '" + name + "' has a Thickness of " + thickness + "\n" +
				"Thickness must be greater than 1.\n\n" +
				"Temporarily setting Thickness to: 1\n" +
				"Please fix the Thickness value for this Boundary in the Plugin Command or Script call."
			);
			thickness = 1;
		}
		
		if (eventId !== -1 && (centerx >= 0 || centery >= 0)) {
			if (dev) alert(
				"⚡ 𝗥𝗶𝘁𝘁𝗲𝗿_𝗘𝘃𝗲𝗻𝘁𝗦𝗽𝗮𝘄𝗻𝗲𝗿.𝗷𝘀 : 𝗕𝗼𝘂𝗻𝗱𝗮𝗿𝘆 𝗖𝗿𝗲𝗮𝘁𝗶𝗼𝗻 : 𝗪𝗔𝗥𝗡𝗜𝗡𝗚! ⚡\n" +
				"🡺 Invalid Target for Boundary! 🡸\n\n" +
				"Boundary Named: '" + name + "' has a Target of " + eventId + " but also has values set for Center X or Center Y\n" +
				"When creating a Boundary on an x, y location please use eventId -1\n" +
				"When creating a Boundary around the Player please use eventId 0, and leave CenterX and CenterY Undefined.\n" +
				"When creating a Boundary around an Event please use Events Id #"
			);
		}
		
		if (abort) return;
		Ritter_Boundary_createBoundary.call(this, width, height, thickness, name, eventId, expandBy, centerx, centery);
		const boundary = this.getBoundary(name);
		boundary.maxEvents = maxEvents;
		boundary.boundaryEvents = [];
		boundary.savedEvents = []; // each index will represent a mapId
		//boundary.savedEvents.map = [];
		for (let mId = 1; mId < $MapData.length; mId++) {
			if (!$MapData[mId]) continue;
			const maps = boundary.savedEvents;
			maps[mId] = []; // each index will represent an x value on the map
			//maps[mId].x = [];
			for (x = 0; x < $MapData[mId].width; x++) {
				maps[mId][x] = []; // each index will represent a y value on the map
				//maps[mId].x[x].y = [];
				for (y = 0; y < $MapData[mId].height; y++) {
					maps[mId][x][y] = null;
				}
			}
		}
		$dataBoundary = this._boundaryType;
	};

	/**
     * Add an Auto Handler to an existing Boundary Object.
     * 
     * @param {string} name - Boundary Name.
     * @param {number} spawnMap - Event Object from a Spawn map.
	 * @param {array} maps - Array of maps Boundary is Active on.
	 * @param {string} type - Type of Boundary. "SpawnOn", "SpawnIn", "UnspawnOn", "UnspawnIn".
	 * @param {number} wait - Wait time in frames between calls to Boundary.
	 * @param {boolean} enabled - Whether Boundary is Enabled or Disabled by Default.
	 * @param {array} boundaries - (Optional) Used for Unspawn Boundaries. Array of "Strings" which contain Boundary Names that the boundary may unspawn events from.
	 * 
	 * @returns Returns to Exit Function when Unspawn Boundary is Detected, the Code After is for Spawn Boundaries.
     */
	Ritter.Boundary.addAutoHandler = function(name, spawnMap, maps, type, wait, enabled, boundaries = false) {
		if (!Array.isArray(maps)) maps = [maps];
		const spawnMapData = $MapData[spawnMap];
		const spawnEvents = spawnMapData.events;
		const boundary = this.getBoundary(name);
		
		boundary.waittime = wait;
		boundary.tick = 0;
		boundary.isRunning = false;
		boundary.enabled = enabled;
		boundary.spawnEvents = [];
		boundary.spawnMap = spawnMap;
		boundary.maps = [];
		boundary.type = type;
		boundary.initialized = false;
		if (boundary.enabled) this.activate(boundary.name);
		if (boundary.type.toLowerCase().contains("unspawn")) {
			boundary.BoundaryList = boundaries.length > 0 ? boundaries[0] : [];
			maps = JSON.parse(maps);
			for (let i = 0; i < maps.length; i++) {
				boundary.maps[maps[i]] = [];
			}
			return;
		} 

		for (i = 0; i < spawnEvents.length; i++) {
			if (!spawnEvents[i]) continue;
			if (!spawnEvents[i].BoundaryData.BoundaryList.contains(boundary.name)) continue;
			const event = spawnEvents[i];
			if (!event || !event.BoundaryData.SpawnRegions) continue;
			this.addSpawnEvent(boundary, event);
		}
	};

	/**
     * Initialize SpawnMap Events' BoundaryData.
	 * 
	 * @param {object} object - SpawnMap Data Object.
     */
	 Ritter.Boundary.extractBoundaryData = function(object) {
		const spawnMap = object;
		const events = spawnMap.events;

		for (let i = 1; i < events.length; i++) {
			if (!events[i]) continue;
			if (!!events[i].BoundaryData) continue;
			events[i].BoundaryData = {};
			data = events[i].BoundaryData;
			data.SpawnRegions = [];
			data.MapList = [];
			data.BoundaryList = [];
				
			for (let j = 0; j < events[i].pages[0].list.length; j++) {
				if (events[i].pages[0].list[j].code === 357 
				&& events[i].pages[0].list[j].parameters[0] === "Ritter_EventSpawner"
				&& events[i].pages[0].list[j].parameters[1] === "BoundaryEventSetup") {
					let EventParams = events[i].pages[0].list[j].parameters[3];
					data.SpawnRegions = JSON.parse(EventParams.RegionsList);
					data.MapList = JSON.parse(EventParams.MapList);
					data.BoundaryList = JSON.parse(EventParams.BoundaryList);
					data.Enabled = JSON.parse(EventParams.Enabled);
					data.SavedEvent = JSON.parse(EventParams.Save);
				}
			}
		}
	};

	/**
 	* Update Boundary saved events.
	* Can be used whenever needed with a script call.
 	* Should not be spammed as data only needs to update on
 	* certain conditions such as spawning/unspawning events
 	* and map change.
	* 
	* Event Unspawns on Boundary: Save Data.
	* Event Spawns on Boundary: Remove Save Data.
	* Change Map: Update only Spawned Saved Events. Unspawned ignored.
 	* 
 	* $gameMap.updateBoundarySavedEvents();
 	*/
	Game_Map.prototype.updateBoundarySavedEvents = function() {
		const map = this;
		const boundaries = Ritter.Boundary._boundaryType;
	
		for (var i = 0; i < boundaries.length; i++) {
			if (boundaries[i].boundaryEvents.length > 0) {
				for (let j = 0; j < boundaries[i].boundaryEvents.length; j++) {
					const eventId = boundaries[i].boundaryEvents[j];
					if (map.event(eventId)._eventData.BoundaryData.SavedEvent) {
						if (map.event(eventId)._erased) {
							continue;
						}
						const x = map.event(eventId).x;
						const y = map.event(eventId).y;
						boundaries[i].savedEvents[map._mapId][x][y] = JsonEx.makeDeepCopy(map.event(eventId));
						Ritter.Boundary.storeSelfSwitches(boundaries[i], map.event(eventId));
					}
				}
			}
		}
	};

	/**
	 * First step in restoring an unspawned saved boundary event.
	 * 
	 * @param {object} boundary - Boundary Object.
	 * @param {number} x - X Location of Saved Event To Restore.
	 * @param {number} y - Y Location of Saved Event To Restore.
	 */
	Game_Map.prototype.restoreUnspawnedSavedBoundaryEvent = function(boundary, x, y) {
		if (!(SceneManager._scene instanceof Scene_Map)) return;
		const event = JsonEx.makeDeepCopy(boundary.savedEvents[this._mapId][x][y]);
		boundary.savedEvents[this._mapId][x][y] = null;
		this.respawnSavedBoundaryEvent(event);
	};

	/**
 	* Respawns/Recreates an event using a saved event as a template.
 	* 
 	* @param {object} event - Saved Game_Event Data Object.
 	*/
	 Game_Map.prototype.respawnSavedBoundaryEvent = function(event) {
		const boundary = Ritter.Boundary.getBoundary(event._parentBoundary);
		var nextId = this._unspawnList.length > 0 ? this._unspawnList[0] : this.getFirstNullEvent();
		var savedEvent = JsonEx.makeDeepCopy(event);
		
		this._events[nextId] = JsonEx.makeDeepCopy(savedEvent);
		spawnEvent = this._events[nextId];
		spawnEvent._eventId = nextId;
		$dataMap.events[nextId] = JsonEx.makeDeepCopy(savedEvent._eventData);
		spawnEvent.boundaryDataInit(spawnEvent._eventData);
		this._spawnedList.push(nextId);
		this._unspawnList.splice(0, 1);
		this._lastSpawnEventId = spawnEvent._eventId;
		spawnEvent._parentBoundary = savedEvent._parentBoundary;
		boundary.boundaryEvents.push(spawnEvent._eventId);
		Ritter.Boundary.restoreSelfSwitches(spawnEvent);
		//boundary.savedEvents.map[this._mapId].x[event.x].y[event.y] = null;
		SceneManager._scene._spriteset.spawnedEvent(spawnEvent);
	};
	
	/**
	 * Stores the Self Switch Data of an Event to be applied on respawn.
	 * 
	 * @param {object} boundary - Boundary Object
	 * @param {object} event - Event Object
	 */
	Ritter.Boundary.storeSelfSwitches = function(boundary, event) {
		const letters = ['A', 'B', 'C', 'D'];
		const mapId = $gameMap._mapId;
		const eventId = event._eventId;
		const x = event._x;
		const y = event._y;
		letters.forEach((letter) => {
			let key = [mapId, eventId, letter];
			let value = $gameSelfSwitches.value(key);
			let index = letters.indexOf(letter);
			let eventData = boundary.savedEvents[mapId][x][y]._eventData;
			if (!eventData.selfSwitchData) eventData.selfSwitchData = [];
			
			eventData.selfSwitchData[index] = value;
		});
	}
	
	/**
	 * Restores the Self Switch Data to an Event.
	 * 
	 * @param {object} event - Event Object
	 */
	Ritter.Boundary.restoreSelfSwitches = function(event) {
		const letters = ['A', 'B', 'C', 'D'];
		const eventId = event._eventId;
		const mapId = $gameMap._mapId;
		const values = event._eventData.selfSwitchData;
		for (let i = 0; i < letters.length; i++) {
			let key = [mapId, eventId, letters[i]];
			$gameSelfSwitches.setValue(key, values[i]);
		}
	}

	/**
	 * Initializes a spawned event using its dataMap/SpawnMap data.
	 * 
	 * @param {object} event - $dataMap event data object.
	 */
	Game_Event.prototype.boundaryDataInit = function(event) {
		this._eventData = event;
		this.initialize(this._mapId, this._eventId);
		DataManager.extractMetadata(this.event());
		DataManager.extractMetadata(event);
	};

	/**
     * Turn on a Boundary.
     * 
     * @param {string} name - Boundary Name.
     */
	Ritter.Boundary.activate = function(name) {
		const boundary = this.getBoundary(name);
		if (!boundary) {
			let dev = Ritter.Params.devMode;
			if (dev) alert(
				"⚡ 𝗥𝗶𝘁𝘁𝗲𝗿_𝗘𝘃𝗲𝗻𝘁𝗦𝗽𝗮𝘄𝗻𝗲𝗿.𝗷𝘀 : 𝗕𝗼𝘂𝗻𝗱𝗮𝗿𝘆 𝗔𝗰𝘁𝗶𝘃𝗮𝘁𝗶𝗼𝗻 : 𝗪𝗔𝗥𝗡𝗜𝗡𝗚! ⚡\n" +
				"🡺 Invalid Boundary Name! 🡸\n\n" +
				"Boundary Named: '" + name + "' could not be found.\n" +
				"Failed to Activate Boundary. \n\n" +
				"Did you use the Correct Name or Have you created this Boundary yet?"
			);
			return;
		}
		if (!this.activeBoundaries.contains(boundary.name)) {
			
			boundary.isRunning = true;
			this.activeBoundaries.push(boundary.name);
		}
	};

	/**
     * Turn off an active Boundary.
     * 
     * @param {string} name - Boundary Name.
     * @param {boolean} unspawnAll - Unspawn all events spawned by this boundary.
     */
	Ritter.Boundary.deactivate = function(name, unspawnAll = false) {
		const boundary = this.getBoundary(name);
		if (!boundary) {
			let dev = Ritter.Params.devMode;
			if (dev) alert(
				"⚡ 𝗥𝗶𝘁𝘁𝗲𝗿_𝗘𝘃𝗲𝗻𝘁𝗦𝗽𝗮𝘄𝗻𝗲𝗿.𝗷𝘀 : 𝗕𝗼𝘂𝗻𝗱𝗮𝗿𝘆 𝗗𝗲𝗮𝗰𝘁𝗶𝘃𝗮𝘁𝗶𝗼𝗻 : 𝗪𝗔𝗥𝗡𝗜𝗡𝗚! ⚡\n" +
				"🡺 Invalid Boundary Name! 🡸\n\n" +
				"Boundary Named: '" + name + "' could not be found.\n" +
				"Failed to Deactivate Boundary. \n\n" +
				"Did you use the Correct Name or Have you created this Boundary yet?"
			);
			return;
		}
		boundary.isRunning = false;
		if (this.activeBoundaries.contains(boundary.name)) 
			this.activeBoundaries = this.activeBoundaries.filter(function(name) { return name !== boundary.name });
		if (!!unspawnAll) {
			$gameMap.updateBoundarySavedEvents();
			for (let i = boundary.boundaryEvents.length - 1; i >= 0; i--) {
				if (!boundary.boundaryEvents[i]) continue;
				Ritter.unspawnEvent(boundary.boundaryEvents[i]);
			}
			boundary.boundaryEvents = [];
		}
	};
	
	/**
     * Add a Spawn Events Data to a Boundary Object.
	 * Uses EventId from Spawn Map or an Event Object.
	 * Event Object from Spawn Map Example:
	 *	 $MapData[2].events[eventId]
     * 
     * @param {string} name - Boundary Name or Boundary Object.
     * @param {number} eventid - Event Id or Event Object from a Spawn Map.
     */
	Ritter.Boundary.addSpawnEvent = function(name, eventid) {
		const boundaryData = typeof name === "string" ? this.getBoundary(name) : name;
		if (!boundaryData) {
			let dev = Ritter.Params.devMode;
			if (dev) alert(
				"⚡ 𝗥𝗶𝘁𝘁𝗲𝗿_𝗘𝘃𝗲𝗻𝘁𝗦𝗽𝗮𝘄𝗻𝗲𝗿.𝗷𝘀 : 𝗕𝗼𝘂𝗻𝗱𝗮𝗿𝘆 𝗔𝗱𝗱 𝗦𝗽𝗮𝘄𝗻 𝗘𝘃𝗲𝗻𝘁 : 𝗪𝗔𝗥𝗡𝗜𝗡𝗚! ⚡\n" +
				"🡺 Invalid Boundary Name! 🡸\n\n" +
				"Boundary Named: '" + name + "' could not be found.\n" +
				"Failed to Deactivate Boundary. \n\n" +
				"Did you use the Correct Name or Have you created this Boundary yet?"
			);
			return;
		}
		const spawnMap = boundaryData.spawnMap;
		const event = typeof eventid === "number" ? $MapData[spawnMap].events[eventid] : eventid;
		if (!event) {
			let dev = Ritter.Params.devMode;
			if (dev) alert(
				"⚡ 𝗥𝗶𝘁𝘁𝗲𝗿_𝗘𝘃𝗲𝗻𝘁𝗦𝗽𝗮𝘄𝗻𝗲𝗿.𝗷𝘀 : 𝗕𝗼𝘂𝗻𝗱𝗮𝗿𝘆 𝗔𝗱𝗱 𝗦𝗽𝗮𝘄𝗻 𝗘𝘃𝗲𝗻𝘁 : 𝗪𝗔𝗥𝗡𝗜𝗡𝗚! ⚡\n" +
				"🡺 Could Not Find Spawn Event! 🡸\n\n" +
				"Event Id: '" + eventid + "' on Spawn Map: " + spawnMap + " could not be found.\n" +
				"Failed to Add Spawn Event to Boundary: '" + name + "'. \n\n" +
				"Did you use the Correct Event Id and Spawn Map Id?"
			);
			return;
		}
		const eventId = event.id;
		const region = event.BoundaryData.SpawnRegions;

		for (let i = 0; i < event.BoundaryData.MapList.length; i++) {
			mapId = event.BoundaryData.MapList[i];
			for (let j = 0; j < region.length; j++) {
				if (!boundaryData.maps[mapId]) {
					boundaryData.maps[mapId] = {};
					boundaryData.maps[mapId].mapId = boundaryData.maps[mapId].mapId || mapId;
					boundaryData.maps[mapId].regions = boundaryData.maps[mapId].regions || [];
				}
				if (!boundaryData.maps[mapId].regions[region[j]]) boundaryData.maps[mapId].regions[region[j]] = [];
				if (boundaryData.maps[mapId].regions[region[j]].contains(eventId)) continue;
				boundaryData.maps[mapId].regions[region[j]].push(eventId);
			}
		}
	};

	/**
     * Remove a Spawn Events Data from a Boundary Object.
	 * Uses EventId from Spawn Map or an Event Object.
	 * Event Object from Spawn Map Example:
	 *   $MapData[2].events[eventId]
     * 
     * @param {string} name - Boundary Name or Boundary Object.
     * @param {number} event - Event Id or Event Object to Remove from Boundary.
     */
	Ritter.Boundary.removeSpawnEvent = function(name, event) {
		const boundaryData = typeof name === "string" ? this.getBoundary(name) : name;
		const spawnMap = boundaryData.spawnMap;
		event = typeof event === "number" ? $MapData[spawnMap].events[event] : event;
		const eventId = event.id;
		const region = event.BoundaryData.SpawnRegions;

		for (let i = 0; i < event.BoundaryData.MapList.length; i++) {
			mapId = event.BoundaryData.MapList[i];
			for (let j = 0; j < region.length; j++) {
				if (!boundaryData.maps[mapId]) {
					boundaryData.maps[mapId] = {};
					boundaryData.maps[mapId].mapId = boundaryData.maps[mapId].mapId || mapId;
					boundaryData.maps[mapId].regions = boundaryData.maps[mapId].regions || [];
				}
				if (!boundaryData.maps[mapId].regions[region[j]]) boundaryData.maps[mapId].regions[region[j]] = [];

				if (!boundaryData.maps[mapId].regions[region[j]].contains(eventId)) continue;
				const index = boundaryData.maps[mapId].regions[region[j]].indexOf(eventId);
				boundaryData.maps[mapId].regions[region[j]].splice(index, 1);
			}
		}
	};

	/**
	 * Change the SpawnMap Id that a Target Boundary uses.
	 * Included for no good reason so be careful here.
	 * Valid Events will need Updating.
	 * Probably easier to just create another boundary.
	 * 
	 * @param {string} name - Name of a Boundary.
	 * @param {number} spawnMap - New SpawnMap Id.
	 */
	Ritter.Boundary.changeSpawnMap = function(name, spawnMap) {
		const boundary = this.getBoundary(name);
		boundary.spawnMap = spawnMap;
	};

	const Ritter_Boundary_EditBoundary = Ritter.Boundary.editBoundary;
	/**
	 * Edit the Properties of a Spawner Boundary.
	 * 
	 * @param {string} boundaryName - Name of a Boundary.
	 * @param {string} prop - Boundary Property to Modify.
	 * @param {number} value - New Property Value.
	 */
	Ritter.Boundary.editSpawnerBoundary = function(boundaryName, prop, value) {
		const boundary = Ritter.Boundary.getBoundary(boundaryName);
		const dev = Ritter.Params.devMode;
		if (dev) {
			if (!boundary) {
				alert(
					"⚡ 𝗥𝗶𝘁𝘁𝗲𝗿_𝗘𝘃𝗲𝗻𝘁𝗦𝗽𝗮𝘄𝗻𝗲𝗿.𝗷𝘀 : 𝗕𝗼𝘂𝗻𝗱𝗮𝗿𝘆 𝗘𝗱𝗶𝘁𝗶𝗻𝗴 : 𝗪𝗔𝗥𝗡𝗜𝗡𝗚! ⚡\n" +
					"🡺 Invalid Boundary Name! 🡸\n\n" +
					"Boundary Named: '" + boundaryName + "' could not be found.\n" +
					"Failed to Set Property: '" + prop + "' to Value: " + value + "\n\n" +
					"Did you use the Correct Name or Have you created this Boundary yet?"
				);
				return;
			}
			if (typeof prop !== "string") {
				alert(
					"⚡ 𝗥𝗶𝘁𝘁𝗲𝗿_𝗘𝘃𝗲𝗻𝘁𝗦𝗽𝗮𝘄𝗻𝗲𝗿.𝗷𝘀 : 𝗕𝗼𝘂𝗻𝗱𝗮𝗿𝘆 𝗘𝗱𝗶𝘁𝗶𝗻𝗴 : 𝗪𝗔𝗥𝗡𝗜𝗡𝗚! ⚡\n" +
					"🡺 Invalid Property Value! 🡸\n\n" +
					"Boundary Name: '" + boundaryName + "'.\n" +
					"Failed to Set Property: '" + prop + "' to Value: " + value + "\n\n" +
					"Please only use \"Strings\" for properties."
				);
				return;
			}
			if (!boundary[prop]) {
				alert(
					"⚡ 𝗥𝗶𝘁𝘁𝗲𝗿_𝗘𝘃𝗲𝗻𝘁𝗦𝗽𝗮𝘄𝗻𝗲𝗿.𝗷𝘀 : 𝗕𝗼𝘂𝗻𝗱𝗮𝗿𝘆 𝗘𝗱𝗶𝘁𝗶𝗻𝗴 : 𝗪𝗔𝗥𝗡𝗜𝗡𝗚! ⚡\n" +
					"🡺 Invalid Boundary Property! 🡸\n\n" +
					"Boundary Name: '" + boundaryName + "'.\n" +
					"Failed to Set Property: '" + prop + "' to Value: " + value + "\n\n" +
					"Property does not exist. Did you type the property string correctly?"
				);
				return;
			}
			if (prop.toLowerCase() === "name") {
				alert(
					"⚡ 𝗥𝗶𝘁𝘁𝗲𝗿_𝗘𝘃𝗲𝗻𝘁𝗦𝗽𝗮𝘄𝗻𝗲𝗿.𝗷𝘀 : 𝗕𝗼𝘂𝗻𝗱𝗮𝗿𝘆 𝗘𝗱𝗶𝘁𝗶𝗻𝗴 : 𝗪𝗔𝗥𝗡𝗜𝗡𝗚! ⚡\n" +
					"🡺 Invalid Boundary Property " + prop + "! 🡸\n\n" +
					"Boundary Name: '" + boundaryName + "'.\n" +
					"Failed to Set Property: '" + prop + "' to Value: " + value + "\n\n" +
					"No changing Boundary Names!"
				);
				return;
			}
			if (isNaN(value)) {
				alert(
					"⚡ 𝗥𝗶𝘁𝘁𝗲𝗿_𝗘𝘃𝗲𝗻𝘁𝗦𝗽𝗮𝘄𝗻𝗲𝗿.𝗷𝘀 : 𝗕𝗼𝘂𝗻𝗱𝗮𝗿𝘆 𝗘𝗱𝗶𝘁𝗶𝗻𝗴 : 𝗪𝗔𝗥𝗡𝗜𝗡𝗚! ⚡\n" +
					"🡺 Invalid New Value! 🡸\n\n" +
					"Boundary Name: '" + boundaryName + "'.\n" +
					"Failed to Set Property: '" + prop + "' to Value: " + value + "\n\n" +
					"Please only use numbers for values."
				);
				return;
			}
		}
		
		if (prop == "maxEvents") {
			boundary["maxEvents"] = value;
			return;
		}
		else if (prop == "waittime") {
			boundary["waittime"] = value;
			return;
		}
		else if (prop == "expandBy") {
			boundary["expandBy"] = value;
			return;
		}
		
		Ritter_Boundary_EditBoundary.call(this, boundaryName, prop, value)
	};

	const Ritter_Boundary_DeleteBoundary = Ritter.Boundary.deleteBoundary;
	/**
	 * Deactivate the Boundary, Unspawn all Boundary Events, and Delete Boundary Object.
	 * 
	 * @param {string} boundaryName - The Name of a Boundary
	 */
	Ritter.Boundary.deleteBoundary = function(boundaryName) {
		Ritter.Boundary.deactivate(boundaryName, true);
		Ritter_Boundary_DeleteBoundary.call(this, boundaryName);
	};
	
	/**
	 * Deactivate All Active Boundaries and Remove All BoundaryEvent Data.
	 */
	Ritter.Boundary.clearAllBoundaryEvents = function() {
		for (let i = Ritter.Boundary.activeBoundaries.length - 1; i >= 0; i--) {
			Ritter.Boundary.deactivate(Ritter.Boundary.activeBoundaries[i], true);
		}
		for (let i = 0; i < Ritter.Boundary._boundaryType.length; i++) {
			Ritter.Boundary._boundaryType[i].boundaryEvents = [];
			Ritter.Boundary._boundaryType[i].initialized = false;
		}
	};

	/**
	 * Initialize Boundaries During GameMap Event Setup.
	 * Turns On Boundaries that are Enabled on the Current $gameMap
	 */
	Ritter.Boundary.initBoundaries = function() {
		const mapId = $gameMap._mapId;
		const boundaries = Ritter.Boundary._boundaryType;

		for (let i = 0; i < boundaries.length; i++) {
			if (!!boundaries[i].maps[mapId] && boundaries[i].enabled == true) {
				 this.activate(boundaries[i].name);
			}
		}
	}

	/**
	 * Respawns/Restores all saved Events within a boundary when called.
	 * 
	 * @param {object} boundary - Boundary Object
	 */
	Ritter.Boundary.initSavedEvents = function(boundary) {
		const tiles = this.getInBoundaryTiles(boundary);
		if (boundary.type.toLowerCase().contains("unspawn")) return;
		for (let i = 0; i < tiles.length; i++) {
			let tile = tiles[i];
			let x = tile[0];
			let y = tile[1];

			if (!(tile[0] == $gamePlayer.x && tile[1] == $gamePlayer.y)
			&& (tile[0] >= 0 && tile[0] <= $dataMap.width - 1)
			&& (tile[1] >= 0 && tile[1] <= $dataMap.height - 1)
			&& !$gameMap.eventIdXy(tile[0], tile[1])
			&& boundary.boundaryEvents.length < boundary.maxEvents
			&& boundary.savedEvents[$gameMap._mapId][tile[0]][tile[1]] !== null) {
				$gameMap.restoreUnspawnedSavedBoundaryEvent(boundary, x, y);
			}
				
		}
	}
//};

})();
