//==========================================================================
// EliMZ_CharManager.js
//==========================================================================

/*:
@target MZ
@base EliMZ_Book
@orderAfter VisuMZ_1_EventsMoveCore

@plugindesc ♦5.2.0♦ Changes a lot of character's sprite configurations.
@author Hakuen Studio
@url https://hakuenstudio.itch.io/eli-character-manager-for-rpg-maker

@help
▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬
• Rate the plugin! Please, is very important to me ^^
https://hakuenstudio.itch.io/eli-character-manager-for-rpg-maker/rate?source=game

• Terms of Use
https://www.hakuenstudio.com/terms-of-use-5-0-0
▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬▬
============================================================================
Features
============================================================================

● Change the following characters configurations:
• Hue Color, Blend Color, Tone Color.
• Fade In/Out Characters
• Scale, Angle, Rotation, Offset, Skew
• Pattern/Frame

● You can change them by:
• Plugin Commands
• Event Notes
• Event Comments

● Works for:
• Events, Players, Followers, and Vehicles!

============================================================================
How to use
============================================================================

https://docs.google.com/document/d/1nrypK787j7TLfvHKv77X5xxMJSBecPvvBDPCe5V6_Pw/edit?usp=sharing

============================================================================

@command changeTone
@text Change Tone
@desc Change the color tone of a character. Separate each one with a comma or set a range like this: [minId, maxId].

    @arg id
    @text Character Id
    @type text
    @desc -2 = First Follower | -1 = Player | 0 = This event 
    1 >= Event Id | 'boat', 'ship', 'airship' Vehicle name
    @default 0

    @arg duration
    @text Duration
    @type text
    @desc Set here the duration in frames. You can use \V[id] or numbers.
    @default 60

    @arg wait
    @text Wait for finish
    @type boolean
    @desc If set to true, the event interpreter will wait until the command is done.
    @default false
    @parent duration

    @arg color
    @text Tone
    @type text
    @desc You can use Html Color(Opacity/Alpha is always 255), hex color or RGBG(Red, Green, Blue, GRAY)
    Range from -255 to 255.
    @default 0,0,0,0

    @arg canDeleteColorFilter
    @text Auto delete color filter
    @type boolean
    @desc If set to true, the color filter will be deleted when the color change is done.
    @default false

@command changeBlendColor
@text Change Blend Color
@desc Change the blend color of a character. Separate each one with a comma or set a range like this: [minId, maxId].

    @arg id
    @text Character Id
    @type text
    @desc -2 = First Follower | -1 = Player | 0 = This event 
    1 >= Event Id | 'boat', 'ship', 'airship' Vehicle name
    @default 0

    @arg duration
    @text Duration
    @type text
    @desc Set here the duration in frames. You can use \V[id] or numbers.
    @default 60

    @arg wait
    @text Wait for finish
    @type boolean
    @desc If set to true, the event interpreter will wait until the command is done.
    @default false
    @parent duration

    @arg color
    @text Blend color
    @type text
    @desc You can use Html Color(Opacity/Alpha is always 255), hex color or RGBA(Red, Green, Blue, ALPHA)
    Range from -255 to 255.
    @default 0,0,0,0

    @arg canDeleteColorFilter
    @text Auto delete color filter
    @type boolean
    @desc If set to true, the color filter will be deleted when the color change is done.
    @default false

@command changeHue
@text Change Hue
@desc Change the hue color of a character. Separate each one with a comma or set a range like this: [minId, maxId].

    @arg id
    @text Character Id
    @type text
    @desc -2 = First Follower | -1 = Player | 0 = This event 
    1 >= Event Id | 'boat', 'ship', 'airship' Vehicle name
    @default 0

    @arg duration
    @text Duration
    @type text
    @desc Set here the duration in frames. You can use \V[id] or numbers.
    @default 60

    @arg wait
    @text Wait for finish
    @type boolean
    @desc If set to true, the event interpreter will wait until the command is done.
    @default false
    @parent duration

    @arg color
    @text Hue
    @type text
    @desc You can use \V[id] and numbers. From -360 to 360
    @default 0

    @arg canDeleteColorFilter
    @text Auto delete color filter
    @type boolean
    @desc If set to true, the color filter will be deleted when the color change is done.
    @default false

@command fade
@text Change Opacity
@desc Fade In/Out characters opacity. Separate each one with a comma or set a range like this: [minId, maxId].

    @arg id
    @text Character Id
    @type text
    @desc -2 = First Follower | -1 = Player | 0 = This event 
    1 >= Event Id | 'boat', 'ship', 'airship' Vehicle name
    @default 0

    @arg duration
    @text Duration
    @type number
    @desc Set here the duration in frames. You can use \V[id] or numbers.
    @default 60

    @arg wait
    @text Wait for finish
    @type boolean
    @desc If set to true, the event interpreter will wait until the command is done.
    @default false
    @parent duration

    @arg opacity
    @text Target Opacity
    @type text
    @desc You can use \V[id], formulas, and numbers.
    @default 255

@command cmd_offset
@text Change Sprite Offset
@desc Fade offset X and Y to events. Separate each one with a comma or set a range like this: [minId, maxId].

    @arg id
    @text Character Id
    @type text
    @desc -2 = First Follower | -1 = Player | 0 = This event 
    1 >= Event Id | 'boat', 'ship', 'airship' Vehicle name
    @default 0

    @arg operation
    @text Operation
    @type select
    @option Set
    @option Add
    @desc Select if you want to set a value or add a value into the existent one.
    @default Set

    @arg x
    @text Offset X
    @type text
    @desc You can use \V[id], formulas, and numbers.
    @default 0
    @parent operation

    @arg y
    @text Offset Y
    @type text
    @desc You can use \V[id], formulas, and numbers.
    @default 0
    @parent operation

@command cmd_angle
@text Change Angle
@desc Change the angle of the sprite in degrees. Separate each one with a comma or set a range like this: [minId, maxId].

    @arg id
    @text Character Id
    @type text
    @desc -2 = First Follower | -1 = Player | 0 = This event 
    1 >= Event Id | 'boat', 'ship', 'airship' Vehicle name
    @default 0

    @arg operation
    @text Operation
    @type select
    @option Set
    @option Add
    @desc Select if you want to set a value or add a value into the existent one.
    @default Set

    @arg angle
    @text Angle in Degrees
    @type text
    @desc You can use \V[id], formulas, and numbers. From -360 to 360.
    @default 0
    @parent operation

@command cmd_rotation
@text Change Rotation
@desc Change the rotation speed of the sprite in degrees. Separate each one with a comma or set a range like this: [minId, maxId].

    @arg id
    @text Character Id
    @type text
    @desc -2 = First Follower | -1 = Player | 0 = This event 
    1 >= Event Id | 'boat', 'ship', 'airship' Vehicle name
    @default 0

    @arg operation
    @text Operation
    @type select
    @option Set
    @option Add
    @desc Select if you want to set a value or add a value into the existent one.
    @default Set

    @arg rotation
    @text Rotation Speed
    @type text
    @desc You can use \V[id], formulas, and numbers.
    @default 0
    @parent operation

@command cmd_scale
@text Change Scale
@desc Change character scale. Separate each one with a comma or set a range like this: [minId, maxId].

    @arg id
    @text Character Id
    @type text
    @desc -2 = First Follower | -1 = Player | 0 = This event 
    1 >= Event Id | 'boat', 'ship', 'airship' Vehicle name
    @default 0

    @arg operation
    @text Operation
    @type select
    @option Set
    @option Add
    @desc Select if you want to set a value or add a value into the existent one.
    @default Set

    @arg x
    @text Scale X
    @type text
    @desc You can use \V[id], formulas, and numbers. Default is 1.
    @default 1
    @parent operation

    @arg y
    @text Scale Y
    @type text
    @desc You can use \V[id], formulas, and numbers. Default is 1.
    @default 1
    @parent operation

@command cmd_skew
@text Change Skew
@desc Change character skew. Separate each one with a comma or set a range like this: [minId, maxId].

    @arg id
    @text Character Id
    @type text
    @desc -2 = First Follower | -1 = Player | 0 = This event 
    1 >= Event Id | 'boat', 'ship', 'airship' Vehicle name
    @default 0

    @arg operation
    @text Operation
    @type select
    @option Set
    @option Add
    @desc Select if you want to set a value or add a value into the existent one.
    @default Set

    @arg x
    @text Skew X
    @type text
    @desc You can use \V[id], formulas, and numbers. Default is 0.
    @default 0
    @parent operation

    @arg y
    @text Skew Y
    @type text
    @desc You can use \V[id], formulas, and numbers. Default is 0.
    @default 0
    @parent operation

@command cmd_extraZ
@text Change Extra Z Index
@desc Change the rotation speed of the sprite in degrees. Separate each one with a comma or set a range like this: [minId, maxId].

    @arg id
    @text Character Id
    @type text
    @desc -2 = First Follower | -1 = Player | 0 = This event 
    1 >= Event Id | 'boat', 'ship', 'airship' Vehicle name
    @default 0

    @arg operation
    @text Operation
    @type select
    @option Set
    @option Add
    @desc Select if you want to set a value or add a value into the existent one.
    @default Set

    @arg zValue
    @text Extra Z
    @type text
    @desc You can use \V[id], formulas, and numbers(including decimals).
    @default 0
    @parent operation

@command cmd_setPatternAnimation
@text Change Pattern / Pattern Animation
@desc Change the pattern of the character.

    @arg id
    @text Character Id
    @type text
    @desc -2 = First Follower | -1 = Player | 0 = This event 
    1 >= Event Id | 'boat', 'ship', 'airship' Vehicle name
    @default 0

    @arg patternList
    @text Pattern List
    @type combo
    @option None
    @option Auto
    @desc Separate each one with a comma.
    @default Auto

    @arg reverse
    @text Reverse Pattern List
    @type boolean
    @desc If true, it will reverse the pattern list above.
    @default false

    @arg freeze
    @text Freeze Pattern
    @type boolean
    @desc Otionally, set this to true if you for the character to freeze on the last pattern.
    @default false

@command deleteCharColorFilter
@text Delete color filter
@desc Delete a character color filter. Separate each one with a comma or set a range like this: [minId, maxId].

    @arg id
    @text Character Id
    @type text
    @desc 0 Current Event | 1, 2... Event Id
    'boat', 'ship', 'airship' Vehicles | -1 Player | -2 First Follower
    @default 0

@command deleteAllColorFilters
@text Delete all color filters
@desc Delete all characters color filter.

*/

"use strict"

var Eli = Eli || {}
var Imported = Imported || {}
Imported.Eli_CharManager = true

/* ========================================================================== */
/*                                   PLUGIN                                   */
/* ========================================================================== */
{

Eli.CharManager = {

    version: 5.11,
    url: "https://hakuenstudio.itch.io/eli-character-manager-for-rpg-maker",
    parameters: {},
    alias: {},

    initialize(){
        this.initParameters()
        this.initPluginCommands()
    },

    initParameters(){},

    initPluginCommands(){
        const commands = [
            'changeTone', 'changeBlendColor', 'changeHue',
            'fade', 'deleteAllColorFilters','deleteCharColorFilter',
            'cmd_offset', 'cmd_angle', 'cmd_rotation', 'cmd_scale', 'cmd_skew',
            'cmd_extraZ', 'cmd_setPatternAnimation'
        ]
        Eli.PluginManager.registerCommands(this, commands)
    },

    param(){
        return this.parameters
    },

    changeTone(args){
        const parameters = this.processColorParameters(args)
        const color = Eli.ColorManager.getRgbForTone(args.color)
        this.processColorChange("setToneColor", color, parameters)
    },

    changeBlendColor(args){
        const parameters = this.processColorParameters(args)
        const color = Eli.ColorManager.getRgbForBlend(args.color)
        this.processColorChange("setBlendColor", color, parameters)
    },

    changeHue(args){
        const parameters = this.processColorParameters(args)
        const color = Number(args.color)
        this.processColorChange("setHueColor", color, parameters)
    },

    processColorParameters(args){
        const canDeleteColorFilter = args.canDeleteColorFilter === "true"
        const duration = Number(Eli.Utils.convertEscapeVariablesOnly(args.duration))
        const ids = Eli.PluginManager.createRangeOfNumbers(args.id)
        const wait = args.wait === "true"

        return [canDeleteColorFilter, duration, ids, wait]
    },

    fade(args){
        const duration = Number(Eli.Utils.convertEscapeVariablesOnly(args.duration))
        const ids =  Eli.PluginManager.createRangeOfNumbers(args.id)
        const targetOpacity = Number(Eli.Utils.convertEscapeVariablesOnly(args.opacity))
        const wait = args.wait === "true"

        for(const id of ids){
            const charId = Eli.Utils.needEval(id)
            const character = Eli.Utils.getMapCharacter(charId)

            character.setFade(targetOpacity, duration)
        }

        if(wait){
            Eli.PluginManager.currentInterpreter.wait(duration)
        }
    },

    deleteAllColorFilters(){
        $gameMap.events().forEach(item => item.deleteColorFilter())
        $gamePlayer.deleteColorFilter()
        $gamePlayer.followers()._data.forEach(item => item.deleteColorFilter())
        $gameMap.vehicles().forEach(item => item.deleteColorFilter())
    },

    deleteCharColorFilter(args){
        const ids = Eli.PluginManager.createRangeOfNumbers(args.id)

        for(const id of ids){
            const character = Eli.Utils.getMapCharacter(id)

            if(character){
                character.deleteColorFilter()
            }
        }
    },

    cmd_offset(args){
        const [ids, x, y] = this.createIdsAndXY(args)
        const isSetOperation = args.operation === "Set"

        for(const id of ids){
            const character = Eli.Utils.getMapCharacter(id)

            if(character){

                if(isSetOperation){
                    character.setCoordOffset(x, y)
                }else{
                    character.addCoordOffset(x, y) 
                }
            }
        }
    },

    cmd_angle(args){
        const ids = Eli.PluginManager.createRangeOfNumbers(args.id)
        const angle = Number(Eli.Utils.processEscapeVarOrFormula(args.angle))

        for(const id of ids){
            const character = Eli.Utils.getMapCharacter(id)

            if(character){

                if(isSetOperation){
                    character.setAngle(angle)
                }else{
                    character.addAngle(angle) 
                }
            }
        }
    },

    cmd_rotation(args){
        const ids = Eli.PluginManager.createRangeOfNumbers(args.id)
        const rotationSpeed = Number(Eli.Utils.processEscapeVarOrFormula(args.rotation))
        const isSetOperation = args.operation === "Set"

        for(const id of ids){
            const character = Eli.Utils.getMapCharacter(id)

            if(character){

                if(isSetOperation){
                    character.setAngleRotationSpeed(rotationSpeed)
                }else{
                    character.addAngleRotationSpeed(rotationSpeed) 
                }
            }
        }
    },

    cmd_scale(args){
        const [ids, x, y] = this.createIdsAndXY(args)
        const isSetOperation = args.operation === "Set"

        for(const id of ids){
            const character = Eli.Utils.getMapCharacter(id)
            
            if(character){

                if(isSetOperation){
                    character.setScale(x, y)
                }else{
                    character.addScale(x, y) 
                }
            }
        }
    },

    cmd_skew(args){
        const [ids, x, y] = this.createIdsAndXY(args)
        const isSetOperation = args.operation === "Set"

        for(const id of ids){
            const character = Eli.Utils.getMapCharacter(id)

            if(character){

                if(isSetOperation){
                    character.setSkew(x, y)
                }else{
                    character.addSkew(x, y) 
                }
            }

        }
    },

    cmd_extraZ(args){
        const ids = Eli.PluginManager.createRangeOfNumbers(args.id)
        const zValue = Number(Eli.Utils.processEscapeVarOrFormula(args.zValue))
        const isSetOperation = args.operation === "Set"

        for(const id of ids){
            const character = Eli.Utils.getMapCharacter(id)

            if(character){

                if(isSetOperation){
                    character.setExtraZIndex(zValue)
                }else{
                    character.addExtraZIndex(zValue) 
                }
            }
        }
    },

    cmd_setPatternAnimation(args){
        const ids = Eli.PluginManager.createRangeOfNumbers(args.id)
        const reverse = args.reverse === "true"
        const freeze = args.freeze === "true"
        const patternList = args.patternList.split(",").map(item => isNaN(item) ? item : Number(item))

        if(patternList[0] === "None"){
            patternList.pop()
        }

        for(const id of ids){
            const char = Eli.Utils.getMapCharacter(id)

            if(char){
                char.setPatternAnimationList(patternList, freeze, reverse)
            }
        }
    },

    processColorChange(setColorType, color, parameters){
        const [canDeleteColorFilter, duration, ids, wait] = parameters

        for(const id of ids){
            const charId = Eli.Utils.needEval(id)
            const character = Eli.Utils.getMapCharacter(charId)

            if(character) {
                character[setColorType](color, duration)
                character.autoDeleteColorFilter = canDeleteColorFilter
    
                if(!character.getMapSprite()._colorFilter) {
                    character.getMapSprite()._createColorFilter()
                }
            }
        }

        if(wait){
            Eli.PluginManager.currentInterpreter.wait(duration)
        }
    },

    createIdsAndXY(args){
        const ids = Eli.PluginManager.createRangeOfNumbers(args.id)
        const x = Number(Eli.Utils.processEscapeVarOrFormula(args.x))
        const y = Number(Eli.Utils.processEscapeVarOrFormula(args.y))

        return [ids, x, y]
    },
}

const Plugin = Eli.CharManager
const Alias = Eli.CharManager.alias

Plugin.initialize()

/* --------------------------- GAME CHARACTER BASE -------------------------- */
{

Alias.Game_CharacterBase_initMembers = Game_CharacterBase.prototype.initMembers
Game_CharacterBase.prototype.initMembers = function() {
    Alias.Game_CharacterBase_initMembers.call(this)
    this.initCharManageMembers()
}

Game_CharacterBase.prototype.initCharManageMembers = function() {
    this.initFadeMembers()
    this.initColorMembers()
    this.initOffsetMembers()
    this.initAngleMembers()
    this.initScaleMembers()
    this.initSkewMembers()
    this.initExtraZIndex()
    this.initPatternMembers()
}

Game_CharacterBase.prototype.initFadeMembers = function() {
    this.fadeChangeDuration = 0
    this.fadeTargetOpacity = 0
}

Game_CharacterBase.prototype.initColorMembers = function(){
    this.needColorFilter = this.needColorFilter || false
    this.autoDeleteColorFilter = this.autoDeleteColorFilter || false
    this.initToneColors()
    this.initBlendColors()
    this.initHueColors()
}

Game_CharacterBase.prototype.initToneColors = function(){
    this.toneColor = this.toneColor || [0, 0, 0, 0]
    this.toneTargetColor = this.toneTargetColor || [0, 0, 0, 0]
    this.toneChangeDuration = 0
}

Game_CharacterBase.prototype.initBlendColors = function(){
    this.blendColor = this.blendColor || [0, 0, 0, 0]
    this.blendTargetColor = this.blendTargetColor || [0, 0, 0, 0]
    this.blendChangeDuration = 0
}

Game_CharacterBase.prototype.initHueColors = function(){
    this.hueColor = this.hueColor || 0
    this.hueTargetColor = 0
    this.hueChangeDuration = 0
}

Game_CharacterBase.prototype.initOffsetMembers = function() {
    this.coordOffsetX = 0
    this.coordOffsetY = 0
}

Game_CharacterBase.prototype.initAngleMembers = function() {
    this.angle = 0
    this.angleRotationSpeed = 0
}

Game_CharacterBase.prototype.initScaleMembers = function() {
    this.scaleX = 1
    this.scaleY = 1
}

Game_CharacterBase.prototype.initSkewMembers = function() {
    this.skewX = 0
    this.skewY = 0
}

Game_CharacterBase.prototype.initExtraZIndex = function() {
    this.extraZIndex = 0
}

Game_CharacterBase.prototype.initPatternMembers = function() {
    this.patternAnimationList = []
    this.patternFreeze = false
}

Game_CharacterBase.prototype.setFade = function(targetOpacity, duration){
    this.fadeTargetOpacity = targetOpacity
    this.fadeChangeDuration = duration
}

Game_CharacterBase.prototype.setToneColor = function(tone, duration){
    this.toneTargetColor = tone
    this.toneChangeDuration = duration
    this.needColorFilter = true
}

Game_CharacterBase.prototype.setBlendColor = function(blendColor, duration){
    this.blendTargetColor = blendColor
    this.blendChangeDuration = duration
    this.needColorFilter = true
}

Game_CharacterBase.prototype.setHueColor = function(hue, duration){
    this.hueTargetColor = hue
    this.hueChangeDuration = duration
    this.needColorFilter = true
}

Game_CharacterBase.prototype.setCoordOffset = function(x, y){
    this.coordOffsetX = x
    this.coordOffsetY = y
}

Game_CharacterBase.prototype.addCoordOffset = function(x, y){
    this.coordOffsetX += x
    this.coordOffsetY += y
}

Game_CharacterBase.prototype.setAngle = function(angle) {
    this.angle = angle   
}

Game_CharacterBase.prototype.addAngle = function(angle) {
    this.angle += angle   
}

Game_CharacterBase.prototype.setAngleRotationSpeed = function(angle) {
    this.angleRotationSpeed = angle   
}

Game_CharacterBase.prototype.addAngleRotationSpeed = function(angle) {
    this.angleRotationSpeed += angle   
}

Game_CharacterBase.prototype.setScale = function(x, y) {
    this.scaleX = x
    this.scaleY = y
}

Game_CharacterBase.prototype.addScale = function(x, y) {
    this.scaleX += x
    this.scaleY += y  
}

Game_CharacterBase.prototype.setSkew = function(x, y) {
    this.skewX = x
    this.skewY = y
}

Game_CharacterBase.prototype.addSkew = function(x, y) {
    this.skewX += x
    this.skewY += y  
}

Game_CharacterBase.prototype.setExtraZIndex = function(value) {
    this.extraZIndex = value
}

Game_CharacterBase.prototype.addExtraZIndex = function(value) {
    this.extraZIndex += value 
}

Game_CharacterBase.prototype.setFreezePattern = function(value) {
    this.patternFreeze = value
    this._animationCount = 0
}

Game_CharacterBase.prototype.setPatternAnimationList = function(patternArray, freeze = false, reverse = false) {
    if(patternArray[0] === "Auto"){
        this.patternAnimationList = Eli.Array.createProgressiveNumbers(0, this.maxPattern()-1)  
    }else{
        this.patternAnimationList = [...patternArray]
    }

    if(reverse){
        this.patternAnimationList.reverse()
    }

    this.setFreezePattern(freeze)
}

Game_CharacterBase.prototype.updateFade = function() {
    if(this.fadeChangeDuration > 0){
        const d = this.fadeChangeDuration

        this._opacity = (this._opacity * (d - 1) + this.fadeTargetOpacity) / d
        this.fadeChangeDuration--
    }
}

Game_CharacterBase.prototype.updateColorFilter = function(sprite){
    this.updateToneColor(sprite)
    this.updateBlendColor(sprite)
    this.updateHue(sprite)
}

Game_CharacterBase.prototype.updateToneColor = function(sprite){
    if (this.toneChangeDuration > 0) {
        const d = this.toneChangeDuration

        for (let i = 0; i < 4; i++) {
            this.toneColor[i] = (this.toneColor[i] * (d - 1) + this.toneTargetColor[i]) / d
        }
        this.toneChangeDuration--

        sprite.setColorTone(this.toneColor)

        if(this.canAutoDeleteColorFilter(this.toneChangeDuration)){
            this.deleteColorFilter()
        }
    }
}

Game_CharacterBase.prototype.updateBlendColor = function(sprite){
    if (this.blendChangeDuration > 0) {
        const d = this.blendChangeDuration

        for (let i = 0; i < 4; i++) {
            this.blendColor[i] = (this.blendColor[i] * (d - 1) + this.blendTargetColor[i]) / d
        }
        this.blendChangeDuration--

        sprite.setBlendColor(this.blendColor)

        if(this.canAutoDeleteColorFilter(this.blendChangeDuration)){
            this.deleteColorFilter()
        }
    }
}

Game_CharacterBase.prototype.updateHue = function(sprite){
    if (this.hueChangeDuration > 0) {
        const d = this.hueChangeDuration

        this.hueColor = (this.hueColor * (d - 1) + this.hueTargetColor) / d
        this.hueChangeDuration--
        
        sprite.setHue(this.hueColor)
        
        if(this.canAutoDeleteColorFilter(this.hueChangeDuration)){
            this.deleteColorFilter()
        }
    }
}

Game_CharacterBase.prototype.updateAngleRotation = function() {
    if(this.angleRotationSpeed !== 0){
        if(this.angle === 360 || this.angle === -360) this.angle = 0
        this.angle = (this.angle + this.angleRotationSpeed).clamp(-360, 360)
    }
}

Game_CharacterBase.prototype.deleteColorFilter = function(){
    this.needColorFilter = false
    this.hueTargetColor = 0
    this.toneColor = [0, 0, 0, 0]
    this.toneTargetColor = [0, 0, 0, 0]
    this.blendColor = [0, 0, 0, 0]
    this.blendTargetColor = [0, 0, 0, 0]
    this.getMapSprite().deleteColorFilter()
}

Game_CharacterBase.prototype.canAutoDeleteColorFilter = function(duration){
    return this.autoDeleteColorFilter && duration <= 0
}

Alias.Game_CharacterBase_screenZ = Game_CharacterBase.prototype.screenZ
Game_CharacterBase.prototype.screenZ = function() {
    return Alias.Game_CharacterBase_screenZ.call(this) + this.extraZIndex
}

Alias.Game_CharacterBase_updateAnimationCount = Game_CharacterBase.prototype.updateAnimationCount
Game_CharacterBase.prototype.updateAnimationCount = function() {
    if(this.patternAnimationList.length > 0){
        this._animationCount += this.maxPattern() * 0.5
    }else{
        Alias.Game_CharacterBase_updateAnimationCount.call(this)
    }
}

Alias.Game_CharacterBase_updatePattern = Game_CharacterBase.prototype.updatePattern
Game_CharacterBase.prototype.updatePattern = function() {
    if(this.patternAnimationList.length > 0){
        this.updateCustomPattern()
        
    }else if(this.patternFreeze){
        this.updatePatternFreeze()
    }else{
        Alias.Game_CharacterBase_updatePattern.call(this)
    }
}

Game_CharacterBase.prototype.updateCustomPattern = function(){
    const pattern = this.patternAnimationList.shift()
    this._pattern = pattern
}

Game_CharacterBase.prototype.updatePatternFreeze = function(){}

}

/* ------------------------------- GAME EVENT ------------------------------- */
{

Game_Event.prototype.parseMeta_Hue = function(string) {
    const data = string.split(",")
    const color = Number(data[0]) || 0
    const duration = 1

    this.setHueColor(color, duration)

    return [color, duration]
}

Game_Event.prototype.parseMeta_BlendColor = function(string) {
    const color = Eli.ColorManager.getRgbForBlend(Eli.String.removeSpaces(string))
    const duration = 1
    
    this.setBlendColor(color, duration)
    
    return [color, duration]
}

Game_Event.prototype.parseMeta_Tone = function(string) {
    const color = Eli.ColorManager.getRgbForTone(Eli.String.removeSpaces(string))
    const duration = 1

    this.setToneColor(color, duration)
    
    return [color, duration]
}

Game_Event.prototype.parseMeta_Offset = function(string) {
    const [x, y] = string.split(",").map(coord => Number(coord) || 0)
    this.setCoordOffset(x, y)

    return [x, y]
}

Game_Event.prototype.parseMeta_Angle = function(value) {
    const angle = Number(value)
    this.setAngle(angle)

    return angle
}

Game_Event.prototype.parseMeta_Rotation = function(value) {
    const rotation = Number(value)
    this.setAngleRotationSpeed(rotation)

    return rotation
}

Game_Event.prototype.parseMeta_Scale = function(string) {
    const [x, y] = string.split(",").map(coord => Number(coord) || 1)
    this.setScale(x, y)

    return [x, y]
}

Game_Event.prototype.parseMeta_Skew = function(string) {
    const [x, y] = string.split(",").map(coord => Number(coord) || 0)
    this.setSkew(x, y)

    return [x, y]
}

Game_Event.prototype.parseMeta_ExtraZ = function(value) {
    const rotation = Number(value)
    this.setExtraZIndex(rotation)

    return rotation
}

Game_Event.prototype.isCharManagerCommentCommand = function(command) {
    return command.code === 108 && command.parameters[0].toLowerCase().includes("charmanager")
}

Alias.Game_Event_afterSetupPage = Game_Event.prototype.afterSetupPage
Game_Event.prototype.afterSetupPage = function(){
    this.needIterateList = this.needIterateList || this.list().some(cmd => this.isCharManagerCommentCommand(cmd))
    Alias.Game_Event_afterSetupPage.call(this)
}

Alias.Game_Event_onListIteration = Game_Event.prototype.onListIteration
Game_Event.prototype.onListIteration = function(index){
    const aliasIndex = Alias.Game_Event_onListIteration.call(this, index)
    const cmd = this.list()[aliasIndex]

    if(cmd && this.isCharManagerCommentCommand(cmd)){
        this.processCharManagerComment(aliasIndex)
    }

    return aliasIndex
}

Game_Event.prototype.processCharManagerComment = function(aliasIndex){
    let comments = ''
    let nextIndex = aliasIndex + 1

    while(this.list()[nextIndex].code === 408){
        const nextCmd = this.list()[nextIndex]
        comments += nextCmd.parameters[0]
        nextIndex++
    }
    
    if(comments.length > 0){
        this.executeCharManagerComment(comments)
    }
}

Game_Event.prototype.executeCharManagerComment = function(comment){
    const regExp = /<([^<>:]+)(:?)([^>]*)>/g
    
    for(;;){
        const match = regExp.exec(comment)
        
        if(match){      
            const key = match[1]
            const value = match[3]
            const dummy = (arg) => arg
            const func = this[`parseMeta_${key}`].bind(this) || dummy.bind(this)
            func(value)

        }else{
            break
        }
    }
}

}

/* ---------------------------- SPRITE CHARACTER ---------------------------- */
{

Alias.Sprite_Character_initMembers = Sprite_Character.prototype.initMembers
Sprite_Character.prototype.initMembers = function() {
    Alias.Sprite_Character_initMembers.call(this)
    this.pivotAdjust = 0
}

Alias.Sprite_Character_setCharacter = Sprite_Character.prototype.setCharacter
Sprite_Character.prototype.setCharacter = function(character){
    Alias.Sprite_Character_setCharacter.call(this, character)

    if(character.needColorFilter){
        this.refreshColorFilter()
    }
}

Sprite_Character.prototype.refreshColorFilter = function() {
    this.setBlendColor(this._character.blendColor)
    this.setColorTone(this._character.toneColor)
    this.setHue(this._character.hueColor)
}

Alias.Sprite_Character_onTileBitmapLoad = Sprite_Character.prototype.onTileBitmapLoad
Sprite_Character.prototype.onTileBitmapLoad = function() {
    Alias.Sprite_Character_onTileBitmapLoad.call(this)
    this.setPivotAdjustment()
}

Alias.Sprite_Character_onCharacterBitmapLoad = Sprite_Character.prototype.onCharacterBitmapLoad
Sprite_Character.prototype.onCharacterBitmapLoad = function() {
    Alias.Sprite_Character_onCharacterBitmapLoad.call(this)
    this.setPivotAdjustment()
}

Sprite_Character.prototype.setPivotAdjustment = function(){
    this.pivotAdjust = this.patternHeight() / 2 
}

Alias.Sprite_Character_updatePosition = Sprite_Character.prototype.updatePosition
Sprite_Character.prototype.updatePosition = function() {
    Alias.Sprite_Character_updatePosition.call(this)
    this.updateCoordOffset()
}

Sprite_Character.prototype.updateCoordOffset = function(){
    this.x += this._character.coordOffsetX
    this.y += this._character.coordOffsetY
}

Alias.Sprite_Character_updateOther = Sprite_Character.prototype.updateOther
Sprite_Character.prototype.updateOther = function() {
    Alias.Sprite_Character_updateOther.call(this)
    this.updateCharManagerFeatures()
}

Sprite_Character.prototype.updateCharManagerFeatures = function(){
    this.updateFadeOpacity()
    this.updateColors()
    this.updateAngleRotation()
    this.updateAngle()
    this.updateScale()
    this.updateSkew()
}

Sprite_Character.prototype.updateFadeOpacity = function(){
    this._character.updateFade()
}

Sprite_Character.prototype.updateColors = function(){
    this._character.updateColorFilter(this)
}

Sprite_Character.prototype.updateAngleRotation = function(){
    this._character.updateAngleRotation()
}

Sprite_Character.prototype.updateAngle = function(){
    this.angle = this._character.angle
    
    if(this.angle !== 0){
        this.adjustPositionForAngle()
    }else{
        this.pivot.y = 0
    }
}

Sprite_Character.prototype.adjustPositionForAngle = function(){
    this.pivot.y = -(this.pivotAdjust / this.scale.y)
    this.y -= this.pivotAdjust
}

Sprite_Character.prototype.updateScale = function(){
    this.scale.x = this._character.scaleX
    this.scale.y = this._character.scaleY
}

Sprite_Character.prototype.updateSkew = function(){
    this.skew.x = this._character.skewX
    this.skew.y = this._character.skewY
}

Sprite_Character.prototype.deleteColorFilter = function(){
    if(this._colorFilter){
        const index = this.filters.indexOf(this._colorFilter)
        this.filters.splice(index, 1)
        this._colorFilter = null
    }
}

}

}