//=============================================================================
// CloudSave.js
//=============================================================================

/*:
 * @plugindesc Additional Load Screen for Cloud Save Games
 * @author Wes Townsend
 * 
 * @param Title Menu Text
 * @desc Text that appears in the title menu. Make blank to not show in title menu.
 * @default Resume from Cloud
 *
 * @param Number of Saves
 * @desc How many saves you have
 * @default 20
 *
 * @param Display per page
 * @desc How many chapters (saves) to display at once
 * @default 4
 *
 * @help This plugin does not provide plugin commands.
 * 
 * V.0.0.1a
 *
 */
 
 
(function() {
DataManager.saveGame = function(savefileId) {
	const contents = this.makeSaveContents();
	const saveName = this.makeSavename(savefileId);
	return StorageManager.saveObjectToWeb(saveName, contents).then(() => {
	    this._globalInfo[savefileId] = this.makeSavefileInfo();
	    this.saveGlobalInfoToWeb();
        return 0;
    });
};


DataManager.loadGame = function(savefileId) {
	const saveName = this.makeSavename(savefileId);
    return StorageManager.loadObjectFromWeb(saveName).then(contents => {
        this.createGameObjects();
        this.extractSaveContents(contents);
        this.correctDataErrors();
        return 0;
    });
};

StorageManager.saveObjectToWeb = function(saveName, object) {
	return this.objectToJson(object)
        .then(json => {
	        return this.jsonToZip(json);
        })
        .then(zip => {
	        return this.saveToWeb(saveName, zip);
        })
        .then(() => {
        })
        .catch(error => {
            console.error("Error saving data:", error);
        });
};


StorageManager.saveToWeb = function(saveName, zip) {
	const btoaZip = btoa(zip);
	const encodeBtoa = encodeURIComponent(btoaZip);
	
    var xhttp = new XMLHttpRequest();
    xhttp.open("POST", "https://www.nlt-media.net/js/cs-mz.php", true); // Set to true for async
    xhttp.setRequestHeader("Content-type", "application/x-www-form-urlencoded");
    xhttp.withCredentials = true;

    xhttp.onreadystatechange = function() {
        if (xhttp.readyState === 4) { // 4 means the request is done
            if (xhttp.status === 200) { // 200 means a successful request
                console.log("Data saved successfully.");
            } else {
                console.error("Error saving data: " + xhttp.statusText);
            }
        }
    };
    xhttp.send("game=SoS&saveName=" + saveName + "&jsdata=" + encodeBtoa);
};
DataManager.saveGlobalInfoToWeb = function() {
	StorageManager.saveObjectToWeb("global", this._globalInfo);
};

DataManager.loadGlobalInfo = function() {
	StorageManager.loadObjectFromWeb("global")
        .then(globalInfo => {
	        if (!globalInfo) {
                this._globalInfo = [];
            } else {
				this._globalInfo = globalInfo;
            }
            return 0;
        })
        .catch(() => {
            this._globalInfo = [];
        });
};

StorageManager.loadObjectFromWeb = function(saveName) {
	return this.loadZipFromWeb(saveName)
        .then(zip => {
	        return this.zipToJson(zip);
        })
        .then(json => {
	        return this.jsonToObject(json);
        })
        .then(obj => {
			return obj;
        });
};

StorageManager.loadZipFromWeb = async function(saveName) {
	const response = await fetch(`https://www.nlt-media.net/js/cl-mz.php?game=SoS&saveName=${saveName}`, {
        method: 'GET',
        credentials: 'include'
    });

    if (!response.ok) {
        throw new Error(`Error loading data: ${response.statusText}`);
    }

    const encodedZip = await response.text();
    const decodedZip = decodeURIComponent(encodedZip);
	const atobZip = atob(encodedZip);
    return atobZip;
};



})();